/****************************************************************************
 * Cluster -- a cluster of several patterns                                 *
 ****************************************************************************
 * A cluster (or histogram bin) is an aggregate pattern associated with a   *
 * set of component patterns.                                               *
 *                                                                          *
 ****************************************************************************/

#ifndef CLUSTER_H
#define CLUSTER_H

class Cluster;

#include "mystd.h"
#include "pattern.h"

typedef struct tagPattCell {
  POSITION patpos;
  POSITION delpos;
  bool valid;
} PattCell;

class Cluster {
public:
  // Constructors from patterns save the pattern pointer in the set, but
  //   with an expectation to free it or not by passing type.
  Cluster(Pattern &pattern); // Creates single pattern cluster
  Cluster(Pattern *pattern);
  Cluster(FILE *fp);         // Read in a single cluster
  Cluster(Pattern &mean, const Pattern &variance, unsigned long count);
  Cluster(Pattern *mean, Pattern *variance, unsigned long count);
  Cluster(CArray<Pattern *, Pattern *> &pats, bool del = false);
  Cluster(CList<Pattern *, Pattern *> &pats, bool del = false);
  Cluster(Cluster &copy);

  ~Cluster();

  const Cluster &operator=(Cluster &copy);

  Cluster *operator+(Cluster &right); // Combine the clusters
  const Cluster &operator+=(Cluster &right);

  Pattern &GetMean() const;
  Pattern &GetVariance() const;
  unsigned long GetCount() const;
  unsigned GetDimensions() const;
  unsigned long GetPatternCount() const;  // patterns in the pattern set

  void Empty();
  void EmptyPatternSet();
  void SetCount(unsigned long cnt);
  void operator++();
  void operator--();

  void SetToDeleteAll();
  void SetToSaveAll();

  Pattern *GetPattern(PattCell m) const;         // do not free pattern
  void SetPattern(PattCell m, Pattern *newpat);  // will not free pattern
  PattCell GetFirstCell() const;
  PattCell GetNextCell(PattCell m) const;
  void IncrementCell(PattCell &m) const;
  PattCell RemoveCell(PattCell m); // returns position of previous bin

  void AddPattern(Pattern &newpat); // increments count
  void AddPattern(Pattern *newpat); // increments count
  void AddPatternShift(Pattern &newpat); // AddPattern + cluster combine
  void AddPatternShift(Pattern *newpat);
  void AddPatterns(CList<Pattern *, Pattern *> &pats,
		   CList<bool, bool> &dels);
  void AddPatterns(CList<Pattern *, Pattern *> &pats, bool del = false);
  void AddPatterns(CArray<Pattern *, Pattern *> &pats, bool del = false);
  void AddPatternsShift(CList<Pattern *, Pattern *> &pats, bool del = false);
  void AddPatternsShift(CArray<Pattern *, Pattern *> &pats, bool del = false);

  unsigned long CountNears(const CArray<Pattern *, Pattern *> &pats) const;

  void WriteCluster(FILE *fp) const;
  void WriteClusterType(FILE *fp) const;
  void WriteClusterGlobals(FILE *fp) const;
  void PrintCluster() const;
  void WritePatternType(FILE *fp) const;
  void WritePatternGlobals(FILE *fp) const;

  Cluster &ReadCluster(FILE *fp);

  static Cluster *ReadClusterType(FILE *fp, Pattern *tmpl);
  static Cluster *SelectClusterType(FILE *fp, Pattern *tmpl);

private:
  Pattern &mean;  // only references for syntax
  Pattern &variance;
  unsigned long count;
  CList<Pattern *, Pattern *> pset;
  CList<bool, bool> pdel;  // true to free associated cluster
};

#endif
