/****************************************************************************
 * ClusterTree -- Binary-like tree of clusters (cluster at leaves)          *
 ****************************************************************************
 *                                                                          *
 ****************************************************************************/

#include "clustree.h"

ClusterTree::ClusterTree(ClusterTree *newleft, ClusterTree *newright,
			 ClusterTree *newroot) {
  root = newroot;
  left = newleft;
  right = newright;
  edgecount = 0;

  if (left)
    edgecount += left->edgecount;
  if (right)
    edgecount += right->edgecount;
}

ClusterTree::~ClusterTree() {
  if (left)
    delete left;
  if (right)
    delete right;
}

bool ClusterTree::HasRoot() const {
  return (root != NULL);
}

bool ClusterTree::HasLeftTree() const {
  return (left != NULL);
}

bool ClusterTree::HasRightTree() const {
  return (right != NULL);
}

ClusterTree &ClusterTree::GetRoot() {
  assert(root != NULL);

  return *root;
}

ClusterTree &ClusterTree::GetLeftTree() {
  assert(left != NULL);

  return *left;
}

ClusterTree &ClusterTree::GetRightTree() {
  assert(right != NULL);

  return *right;
}

ClusterTree *ClusterTree::SetRoot(ClusterTree *newroot) {
  ClusterTree *temp = root;
  root = newroot;
  return temp;
}

ClusterTree *ClusterTree::SetLeftTree(ClusterTree *newleft) {
  ClusterTree *temp = left;
  left = newleft;
  return temp;
}

ClusterTree *ClusterTree::SetRightTree(ClusterTree *newright) {
  ClusterTree *temp = right;
  right = newright;
  return temp;
}

unsigned long ClusterTree::GetEdgeCount() const {
  return edgecount;
}

bool ClusterTree::IsEdge() const {
  return false;
}

ClusterEdge &ClusterTree::GetEdge() {
  assert(IsEdge());

  return *((ClusterEdge *) this);
}

ClusterTree &ClusterTree::ScaleToTop() {
  ClusterTree *tree = this;

  while (tree->root)
    tree = tree->root;

  return *tree;
}

// Checks by pointers
bool ClusterTree::IsInTree(Pattern &find) {
  if (!root && IsEdge())
    return (&GetEdge().GetPatternOne() == &find ||
	    &GetEdge().GetPatternTwo() == &find);

  ClusterTree &treeroot = ScaleToTop();
  return (treeroot.left->IsInBranch(find) || treeroot.right->IsInBranch(find));
}

bool ClusterTree::IsInBranch(Pattern &find) {
  if (IsEdge())
    return (&GetEdge().GetPatternOne() == &find ||
	    &GetEdge().GetPatternTwo() == &find);

  return (left->IsInBranch(find) || right->IsInBranch(find));
}

bool ClusterTree::IsInTree(ClusterTree &find) {
  return (&ScaleToTop() == &find.ScaleToTop());
}

ClusterTree *ClusterTree::CombineWith(ClusterTree &two) {
  ClusterTree &oldtree1 = ScaleToTop();
  ClusterTree &oldtree2 = two.ScaleToTop();

  if (&oldtree1 == &oldtree2) // Combination unnecessary
    return &oldtree1;

  // Generate new tree
  ClusterTree *newtree = new ClusterTree(&oldtree1, &oldtree2);

  // Point old roots to new tree
  oldtree1.SetRoot(newtree);
  oldtree2.SetRoot(newtree);
  newtree->SetLeftTree(&oldtree1);
  newtree->SetRightTree(&oldtree2);

  return newtree;
}

// Combines the leaves of all subtrees into a single cluster
Cluster *ClusterTree::MakeCluster() {
  ASSERT(left);
  ASSERT(right);
  Cluster *one = left->MakeCluster();
  Cluster *two = right->MakeCluster();
  Cluster *newcls = *one + *two;

  delete one;
  delete two;

  return newcls;
}

/***** ClusterEdge Implementation *******************************************/

ClusterEdge::ClusterEdge(ClusterNode &newone, ClusterNode &newtwo,
			 ClusterTree *root) :
  one(newone), two(newtwo), ClusterTree(NULL, NULL, root) {
  weight = one.GetPattern().CartesianDistance(two.GetPattern());
}

void ClusterEdge::Mark() {
  ClusterTree *onetree, *twotree;

  // Get the appropriate trees (may be tree consisting of just this edge)
  if (one.IsUsed())
    onetree = &one.ScaleToTop();
  else
    onetree = this;

  if (two.IsUsed())
    twotree = &two.ScaleToTop();
  else
    twotree = this;

  // Check if already part of the same graph
  if (onetree == twotree)
    return;  // Unnecessary

  // Mark the edges with the nodes
  one.AddEdge(*this);
  two.AddEdge(*this);

  // Now combine the two graphs
  onetree->CombineWith(*twotree);
}

double ClusterEdge::GetWeight() const {
  return weight;
}

Pattern &ClusterEdge::GetPatternOne() {
  return one.GetPattern();
}

Pattern &ClusterEdge::GetPatternTwo() {
  return two.GetPattern();
}

ClusterNode &ClusterEdge::GetNodeOne() {
  return one;
}

ClusterNode &ClusterEdge::GetNodeTwo() {
  return two;
}

bool ClusterEdge::IsEdge() const {
  return (!left && !right);
}

Cluster *ClusterEdge::MakeCluster() {
  one.GetPattern().DuplicateType();
  Cluster *newcls = new Cluster(one.GetPattern());
  newcls->AddPatternShift(two.GetPattern());

  return newcls;
}

/***** ClusterEdge Implementation *******************************************/

ClusterNode::ClusterNode() {
  pat = NULL;
}

ClusterNode::~ClusterNode() {
  edges.RemoveAll();
}

Pattern &ClusterNode::GetPattern() {
  assert(pat);
  return *pat;
}

void ClusterNode::SetPattern(Pattern &newpat) {
  pat = &newpat;
}

void ClusterNode::AddEdge(ClusterEdge &edge) {
  edges.AddHead(&edge);
}

bool ClusterNode::IsUsed() const {
  return edges.GetCount();
}

ClusterTree &ClusterNode::ScaleToTop() {
  return edges.GetHead()->ScaleToTop();
}

/****** Auxillary Functions *************************************************/

// For sorting edges
int EdgeCompare(const ClusterEdge *edge1, const ClusterEdge *edge2) {
  if (edge1->weight > edge2->weight)
    return 1;
  if (edge1->weight < edge2->weight)
    return -1;
  return 0;
}

int EdgeCompare2(const void *edge1, const void *edge2) {
  return EdgeCompare(*((const ClusterEdge **) edge1),
		     *((const ClusterEdge **) edge2));
}
