/****************************************************************************
 * ClusterTree -- Graph built from edges of patterns, build like a tree     *
 ****************************************************************************
 * Implemented like a tree, where a single graph is a collection of smaller *
 * graphs and those of smaller graphs, to the smallest graph, an edge       *
 * All such graph branches share ClusterNode's.                             *
 *                                                                          *
 ****************************************************************************/

#ifndef CLUSTERTREE_H
#define CLUSTERTREE_H

class ClusterTree;
class ClusterEdge;
class ClusterNode;

#include "mystd.h"
#include <List.h>
#include "cluster.h"
#include "pattern.h"

class ClusterTree {
public:
  ClusterTree(ClusterTree *newleft = NULL, ClusterTree *newright = NULL,
	      ClusterTree *newroot = NULL);
  ~ClusterTree();

  bool HasRoot() const;
  bool HasLeftTree() const;
  bool HasRightTree() const;

  ClusterTree &GetRoot();
  ClusterTree &GetLeftTree();
  ClusterTree &GetRightTree();

  ClusterTree *SetRoot(ClusterTree *newroot);
  ClusterTree *SetLeftTree(ClusterTree *newleft);
  ClusterTree *SetRightTree(ClusterTree *newright);

  unsigned long GetEdgeCount() const;

  virtual bool IsEdge() const;
  ClusterEdge &GetEdge();

  ClusterTree &ScaleToTop();

  bool IsInTree(Pattern &find);
  bool IsInBranch(Pattern &find);
  bool IsInTree(ClusterTree &find);

  ClusterTree *CombineWith(ClusterTree &two);

  virtual Cluster *MakeCluster();

protected:
  ClusterTree *root;
  ClusterTree *left;
  ClusterTree *right;

  unsigned long edgecount;
};

// An edge is first created, but only later placed in a graph by marking it
class ClusterEdge : public ClusterTree {
  friend int EdgeCompare(const ClusterEdge *edge1, const ClusterEdge *edge2);
public:
  ClusterEdge(ClusterNode &newone, ClusterNode &newtwo,
	      ClusterTree *root = NULL);

  void Mark();

  double GetWeight() const;
  Pattern &GetPatternOne();
  Pattern &GetPatternTwo();
  ClusterNode &GetNodeOne();
  ClusterNode &GetNodeTwo();

  virtual bool IsEdge() const;

  virtual Cluster *MakeCluster();

private:
  double weight;
  ClusterNode &one;
  ClusterNode &two;
};

class ClusterNode {
public:
  ClusterNode();
  ~ClusterNode();

  Pattern &GetPattern();
  void SetPattern(Pattern &newpat);

  void AddEdge(ClusterEdge &edge);  // Usually called when an edge is marked

  bool IsUsed() const;

  ClusterTree &ScaleToTop();

private:
  Pattern *pat;
  CList<ClusterEdge *, ClusterEdge *> edges;
};

int EdgeCompare(ClusterEdge *edge1, ClusterEdge *edge2);
int EdgeCompare2(const void *edge1, const void *edge2);

#endif
