/****************************************************************************
 * GausImage -- Pyramid of images, successively reduced by Gaussian filters *
 ****************************************************************************
 *                                                                          *
 ****************************************************************************/

#include "gausimage.h"
#include "image.h"

/***** GausImage Implementation *********************************************/

// Gaussian Pyramid by applying low-pass Gaussian filter on each level
// Uses pointer to base level of image, allocates all memory for others
GausImage::GausImage(Image &img) :
  source(img), GausStructure<GausImageLevel>() {

  unsigned k = 0;
  for (Image *currimg = &img;
       currimg->GetRowCount() > 1 && currimg->GetColumnCount() > 1;
       k++, currimg = currimg->Reduce()) {
    AddLevel(new GausImageLevel(*currimg));
  }

  SetLevelCount(k);
}

GausImage::GausImage(unsigned k) :
  source(*(new Image(0, 0))), GausStructure<GausImageLevel>(k) {
}

// Frees the memory for pyramid images
// Leaves source image untouched, because this was the image passed
//   to constructor (and so was not allocated by it)
GausImage::~GausImage() {
  for (unsigned k = 1; k < levcount; k++)
    delete &GetLevel(k).GetImage();
}

CArray<Pattern *, Pattern *> *GausImage::MakePatterns() const {
  CArray<Pattern *, Pattern *> *newpats
    = new CArray<Pattern *, Pattern *>;

  newpats->SetSize(GetLevel(0).GetImage().GetPixelCount());

  unsigned rows = GetLevel(0).GetImage().GetRowCount();
  unsigned cols = GetLevel(0).GetImage().GetColumnCount();
  for (unsigned r = 0; r < rows; r++)
    for (unsigned c = 0; c < cols; c++)
      newpats->Add(new GausPattern(*this, r, c));

  return newpats;
}

CArray<LocPattern *, LocPattern *> *GausImage::MakeLocPatterns() const {
  CArray<Pattern *, Pattern *> *pats = MakePatterns();
  CArray<LocPattern *, LocPattern *> *locpats =
    new CArray<LocPattern *, LocPattern *>;

  locpats->SetSize(pats->GetSize());

  unsigned rows = GetLevel(0).GetImage().GetRowCount();
  unsigned cols = GetLevel(0).GetImage().GetColumnCount();
  for (unsigned r = 0; r < rows; r++)
    for (unsigned c = 0; c < cols; c++)
      locpats->Add(new LocPattern(pats->GetAt(r * cols + c), r, c));

  return locpats;
}

/***** GausImageLevel Implementation ****************************************/

GausImageLevel::GausImageLevel(Image &img) :
  image(img) {
}

Image &GausImageLevel::GetImage() const {
  return image;
}
