/****************************************************************************
 * GausStructure -- parent class for leveled data structures from Gaussians *
 ****************************************************************************
 *                                                                          *
 ****************************************************************************/

#ifndef GAUSSTRUCT_H
#define GAUSSTRUCT_H

#define GAUSSIAN_COMPRS .5

template<class T> class GausStructure;
template<class T> class GausStructureLevel;

#include "mystd.h"

template<class T>  // class T derives from GausStructureLevel
class GausStructure {
public:
  GausStructure(unsigned k = 0);
  virtual ~GausStructure();  // need to free any child class memory

  unsigned GetLevelCount() const;
  unsigned GetFilledLevels() const;
  void SetLevelCount(unsigned k);

  void AddLevel(T *lev);

  T &GetLevel(unsigned k) const;
  void SetLevel(unsigned k, T *lev);

protected:
  // array built of pointers for combining of levels
  CArray<T *, T *> levels;
  unsigned levcount;  // specified number of levels     levcount >= nextlev
  unsigned nextlev;   // top level not yet filled in
};

template<class T>  // class T derives from GausStructureLevel
class GausStructureLevel {
public:
  GausStructureLevel();
  GausStructureLevel(T &abv);
  virtual ~GausStructureLevel();  // need to free any child class memory

  T &GetAbove() const;
  void SetAbove(T &abv);

protected:
  T *above;
};

/****************************************************************************
 * GausStructure -- parent class for leveled data structures from Gaussians *
 ****************************************************************************
 *                                                                          *
 ****************************************************************************/

#include "gausstruct.h"

/***** GausStructure Implmentation ******************************************/

template<class T>
GausStructure<T>::GausStructure(unsigned k) {
  levcount = k;
  nextlev = 0;
  levels.SetSize(k);  // just allocate memory
}

template<class T>
GausStructure<T>::~GausStructure() {
  for (unsigned k = 0; k < nextlev; k++)
    delete levels[k];
}

template<class T>
unsigned GausStructure<T>::GetLevelCount() const {
  return levcount;
}

template<class T>
unsigned GausStructure<T>::GetFilledLevels() const {
  return nextlev;
}

template<class T>
void GausStructure<T>::SetLevelCount(unsigned k) {
  levcount = k;
  if (nextlev > k)
    nextlev = k;
  levels.SetSize(k);
}

template<class T>
void GausStructure<T>::AddLevel(T *lev) {
  levels.Add(lev);

  if (nextlev > 0)
    levels[nextlev - 1]->SetAbove(*lev);

  if (levcount == nextlev)
    levcount++;
  nextlev++;
}

template<class T>
T &GausStructure<T>::GetLevel(unsigned k) const {
  assert(k < nextlev);
  return *((T *) levels[k]);
}

template<class T>
void GausStructure<T>::SetLevel(unsigned k, T *lev) {
  assert(k < nextlev);
  if (levels[k])
    delete levels[k];

  levels.SetAt(k, lev);

  if (k > 0)
    levels[k - 1]->SetAbove(*lev);
}

/***** GausStructureLevel Implmentation *************************************/

template<class T>
GausStructureLevel<T>::GausStructureLevel() {
  above = NULL;
}

template<class T>
GausStructureLevel<T>::GausStructureLevel(T &abv) {
  SetAbove(abv);
}

template<class T>
GausStructureLevel<T>::~GausStructureLevel() {
}

template<class T>
T &GausStructureLevel<T>::GetAbove() const {
  return *above;
}

template<class T>
void GausStructureLevel<T>::SetAbove(T &abv) {
  above = &abv;
}

#endif
