/*****************************************************************************
 * Image -- functions for using images (ppm files) in Viola's algorithm      *
 *****************************************************************************
 *                                                                           *
 ****************************************************************************/

#include "image.h"
#include "gausimage.h"

// Posible, safe pixel-accessing macros
// Kill boundaries
#define bimg(r, c) pixels[(r < 0 || r >= rows) ? 0 : r][(c < 0 || c >= cols) ? 0 : c]

// Wrap around boundaries
// #define bimg(r, c) pixels[(r < 0 || r >= rows) ? (rows - (r) - 1) : r][(c < 0 || c >= cols) ? (cols - (c) - 1) : c]

// Block boundaries
// #define bimg(r, c) pixels[(r < 0) ? 0 : ((r >= rows) ? (rows - 1) : r)][(c < 0) ? 0 : ((c >= cols) ? (cols - 1) : c)]

// Gaussian Kernal for image reduction
static float w[3] = {.4, .25, .05};

Image::Image(char *filename) {
  FILE *fp = fopen(filename, "rb");

  assert(fp);

  char line[200];

  // Top line should be PPM type
  if (strcmp(fgets(line, 200, fp), "P6\n")) {  // error
    rows = 0;
    cols = 0;
    pixels = NULL;
    return;
  }

  // Remove comments
  while (*(fgets(line, 200, fp)) == '#');

  // Collect dimensions
  cols = atoi(strtok(line, " \n"));
  rows = atoi(strtok(NULL, " \n"));

  // Allocate the image
  pixels = new Pixel*[rows];
  for (unsigned r = 0; r < rows; r++)
    pixels[r] = new Pixel[cols];

  fgets(line, 200, fp);  // Remove "255" line

  // Read in the pixels
  for (unsigned r = 0; r < rows; r++)
    for (unsigned c = 0; c < cols; c++)
      pixels[r][c] = (fgetc(fp) + fgetc(fp) + fgetc(fp)) / 3;

  fclose(fp);
}

Image::Image(unsigned nrows, unsigned ncols) {
  rows = nrows;
  cols = ncols;

  pixels = new Pixel*[rows];
  for (unsigned r = 0; r < rows; r++) {
    pixels[r] = new Pixel[cols];
    for (unsigned c = 0; c < cols; c++)
      pixels[r][c] = 0;
  }
}

Image::~Image() {
  for (unsigned r = 0; r < rows; r++)
    delete pixels[r];

  delete pixels;
}

unsigned Image::GetRowCount() const {
  return rows;
}

unsigned Image::GetColumnCount() const {
  return cols;
}

unsigned long Image::GetPixelCount() const {
  return rows * cols;
}

int Image::SaveImage(const char *filename) const {
  FILE *fp = fopen(filename, "wb");

  // Set up header
  fprintf(fp, "P6\n");
  fprintf(fp, "# CREATOR: VIOLA.C\n");
  fprintf(fp, "%d %d\n255\n", cols, rows);

  // Write out pixels
  for (unsigned r = 0; r < rows; r++)
    for (unsigned c = 0; c < cols; c++)
      fprintf(fp, "%c%c%c", pixels[r][c], pixels[r][c], pixels[r][c]);

  fclose(fp);

  return 0;
}

Pixel Image::GetPixel(unsigned r, unsigned c) const {
  return bimg(r, c);
}

void Image::SetPixel(unsigned r, unsigned c, Pixel pix) {
  if (r < 0 || r >= rows || c < 0 || c >= cols)
    return;

  pixels[r][c] = pix;
}

// Returns image reduced by GAUSSIAN_COMPRS, blurred with a Gaussian filter
Image *Image::Reduce() const {
  Image *gaus = new Image(rows * GAUSSIAN_COMPRS, cols * GAUSSIAN_COMPRS);

  for (unsigned r = 0; r < gaus->rows; r++)
    for (unsigned c = 0; c < gaus->cols; c++)
      (gaus->pixels)[r][c] = GaussianAt((r / GAUSSIAN_COMPRS),
					(c / GAUSSIAN_COMPRS));

  return gaus;
}

Image *Image::Expand() const {
  Image *gaus = new Image(rows / GAUSSIAN_COMPRS, cols / GAUSSIAN_COMPRS);

  for (unsigned r = 0; r < gaus->rows; r++)
    for (unsigned c = 0; c < gaus->cols; c++)
      (gaus->pixels)[r][c] = bimg((unsigned) (r * GAUSSIAN_COMPRS),
				  (unsigned) (c * GAUSSIAN_COMPRS));

  return gaus;
}

// Applies the filter to the pixels surrounding (r, c), returns the response
float Image::CalcFeatureAt(unsigned r, unsigned c, float filter[3][3]) const {
  float resp = 0;

  // occasionally the reduction from GAUSSIAN_COMPRS will reduce the total
  // rows or columns faster than the specific row or column number
  if (r >= rows)
    r = rows - 1;
  if (c >= cols)
    c = cols - 1;

  for (char i = 0; i <= 2; i++)
    for (char j = 0; j <= 2; j++)
      resp += filter[i][j] * bimg(r + i - 1, c + j - 1);

  return resp;
}

// Calculates the 5-by-5 window Gaussian approximation abount a pixel
Pixel Image::GaussianAt(unsigned r, unsigned c) const {
  float pix = 0; // Use float until convert to Pixel at end

  for (char i = -2; i <= 2; i++)
    for (char j = -2; j <= 2; j++)
      pix += w[abs(i)] * w[abs(j)] * bimg(r + i, c + j);

  return (Pixel) pix;
}
