/****************************************************************************
 * Model -- All the information for a complete probability model            *
 ****************************************************************************
 *                                                                          *
 ****************************************************************************/

#include "model.h"
#include "system.h"
#include "histogram.h"

#define PROB_INCLASS    .5

Model::Model() {
}

Model::Model(Histogram *in) {
  hists.AddHead(in);
}

Model::~Model() {
  POSITION pos = hists.GetHeadPosition();
  while (pos)
    delete hists.GetNext(pos);
}

Histogram &Model::GetModelHistogram() {
  if (hists.GetCount() < 1)
    hists.AddHead(new Histogram());
  return *(hists.GetHead());
}

unsigned Model::GetHistogramCount() {
  return hists.GetCount();
}

void Model::SetHistogramCount(unsigned count) {
  while (hists.GetCount() < count)
    hists.AddTail(new Histogram());
}

Histogram &Model::GetHistogram(unsigned id) {
  POSITION pos = hists.GetHead();
  while (id--)
    hists.GetNext(pos);
  return *hists.GetAt(pos);
}

void Model::AddHistogram(Histogram *newhist) {
  hists.AddTail(newhist);
}

void Model::AddHistogram() {
  hists.AddTail(new Histogram());
}

// Probability by comparing the histogram of the test image relative to the
//   model histogram to the histogram of the test image relative to the
//   out-of-class histogram
double Model::CalcEntropy(const CArray<Pattern *, Pattern *> &pats) {
  double totalent = 0;

  POSITION pos = hists.GetHeadPosition();
  while (pos) {  
    Histogram modeltest(*hists.GetAt(pos));  // Creates new clusters
    modeltest.TestHistogram(pats);
    totalent += hists.GetNext(pos)->EntropyByClass(modeltest);
  }

  return totalent;
}

// Same a EntropyProb, but uses ChiSqr
// Calculates the [multiple-image] ChiSqr metric
// Chi^2 = Sum_i((Sqrt(Sum_k(ModelImageAvg_k) / TestImageAvg) ModelBin_i -
//                Sqrt(TestImageAvg / Sum_k(ModelImageAvg_k)) TestBin_i)^2
//               / ModelBin_i)
double Model::CalcChiSqr(const CArray<Pattern *, Pattern *> &pats) {
  // For non-multiple image models
  if (GetHistogramCount() == 1) {
    Histogram modeltest(*hists.GetHead());  // Creates new clusters
    modeltest.TestHistogram(pats);
    return hists.GetHead()->ChiSqrByClass(modeltest);
  } else {
    double chisqr = 0.0;

    // Calculate scale factor for multiple images
    double scale;
    POSITION pos = hists.GetHeadPosition();
    while (pos)
      scale += hists.GetNext(pos).GetPatternCount();
    scale = sqrt(scale / pats.GetSize());

    // Do several chisqr tests and combine
    pos = hists.GetHeadPosition();
    while(pos) {
      Histogram modeltest(*hists.GetAt(pos));
      modeltest.TestHistogram(pats);
      chisqr += GetNext(pos).MultipleChiSqrByClass(modeltest, scale);
    }

    return chisqr;
  }
}

// Probability based on checking each pattern against each histogram 
// TODO: Make this general for arbitrary numbers of histograms
double Model::CalcPattern(const CArray<Pattern *, Pattern *> &pats) {
  unsigned long count = pats.GetSize();
  double prob = 0.0;

  for (unsigned long m = 0; m < count; m++)
    prob += System::BayesRule(hists.GetHead()->ProbByPattern(*pats[m]),
			      hists.GetTail()->ProbByPattern(*pats[m]));

  return prob / (double) count;
}
