/****************************************************************************
 * Pattern -- Individual pattern vector class for Viola's texture algorithm *
 ****************************************************************************
 *                                                                          *
 ****************************************************************************/

#include "pattern.h"
#include "histfile.h"

#define PATTERN_DT      .2
#define RESPONSE_DT     .9

// Various Response kernals
// For now, kernals are always 3x3 normalized arrays
static float gaus[3][3] = {{.0625, .125, .0625},
			   {.125, .25, .125},
			   {.0625, .125, .0625}};
static float vbar[3][3] = {{.0625, -.125, .0625},
			   {.125, -.25, .125},
			   {.0625, -.125, .0625}};
static float hbar[3][3] = {{.0625, .125, .0625},
			   {-.125, -.25, -.125},
			   {.0625, .125, .0625}};
static float vedg[3][3] = {{.125, 0, -.125},
			   {.25, 0, -.25},
			   {.125, 0, -.125}};
static float hedg[3][3] = {{.125, .25, .125},
			   {0, 0, 0},
			   {-.125, -.25, -.125}};

#define PR_GAUS 0
#define PR_VBAR 1
#define PR_HBAR 2
#define PR_VEDG 3
#define PR_HEDG 4

/***** Pattern Implmentation ************************************************/

Pattern::Pattern(unsigned d) {
  dims = d;
}

unsigned Pattern::GetDimensions() const {
  return dims;
}

void Pattern::Zero() {
  for (unsigned i = 0; i < dims; i++)
    SetResponse(i, 0.0);
}

// Converts the pattern to float by summing responses
float Pattern::SumReduce() const {
  float total = 0.0;

  for (unsigned i = 0; i < dims; i++)
    total += GetResponse(i);

  return total;
}

float Pattern::SumSqrReduce() const {
  float total = 0.0;

  for (unsigned i = 0; i < dims; i++)
    total += sqr(GetResponse(i));

  return total;
}

void Pattern::RandomShift(float factor) {
  for (unsigned i = 0; i < dims; i++)
    SetResponse(i, factor * (2.0 * frand() - 1.0) * GetResponse(i));
}

Pattern *Pattern::Sqr() const {
  Pattern *newpat = Duplicate();

  for (unsigned i = 0; i < dims; i++)
    newpat->SetResponse(i, sqr(GetResponse(i)));

  return newpat;
}

// Calculates sqrt(delta1^2 + delta2^2 + ... + deltan^2) (Cartesian distance)
float Pattern::CartesianDistance(const Pattern &two) const {
  float dist = 0.0;

  assert(dims == two.dims);

  for (unsigned i = 0; i < dims; i++)
    dist += sqr(GetResponse(i) - two.GetResponse(i));

  return sqrt(dist);
}  

// Applies ResponseDistance to each feature in each pattern
bool Pattern::NearByResponse(const Pattern &two) const {
  assert(dims == two.dims);

  for (unsigned i = 0; i < dims; i++)
    if (ResponseDistance(GetResponse(i), two.GetResponse(i)) > RESPONSE_DT)
      return false;

  return true;  // Passed all tests
}

// This is another attempt at calculating perceptual distance
// Use with NearByResponse
float Pattern::ResponseDistance(float respone, float resptwo) const {
  return fabs(respone - resptwo) / (fabs(respone) + fabs(resptwo) + 1.0);
}

bool Pattern::NearByPattern(const Pattern &two, float Znorm) const {
  return (PatternDistance(two) / Znorm <= PATTERN_DT);
}

// Sum (one[i] - two[i])^2 -- unnormalized
// Note: does not use variance
// Algorithm from the synthesis papers
// Use with NearByPattern
float Pattern::PatternDistance(const Pattern &two) const {
  float total = 0.0;

  assert(dims == two.dims);

  for (unsigned i = 0; i < dims; i++)
    total += sqr(GetResponse(i) - two.GetResponse(i));

  return total;
}

Pattern *Pattern::SelectPatternType(FILE *fp) {
  char filetype[128];

  fgets(filetype, 128, fp);
  filetype[strlen(filetype) - 1] = '\0'; // Drop newline

  if (!strcmp(GAUS_PATTERN_TYPE, filetype))
    return GausPattern::ReadPatternType(fp);
  if (!strcmp(LOC_PATTERN_TYPE, filetype))
    return LocPattern::ReadPatternType(fp);
}

ePatternType Pattern::GetPatternType() const {
  return PatternBaseType;
}

/***** GausPattern Implementation *******************************************/

GausPattern::GausPattern(unsigned i) :
  Pattern(i), GausStructure<GausPatternLevel>(i / FEATURES + 1) {

  assert(!(i % FEATURES));

  i /= FEATURES;
  i++;

  while (i--)
    AddLevel(new GausPatternLevel);
}

// Copy constructor
GausPattern::GausPattern(const GausPattern &copy) :
  Pattern(copy.dims), GausStructure<GausPatternLevel>(copy.levcount) {

  for (unsigned k = 0; k < copy.levcount; k++)
    AddLevel(new GausPatternLevel(copy.GetLevel(k)));
}

// Scales pyramid to find response at each level
GausPattern::GausPattern(const GausImage &gimg, unsigned r, unsigned c) :
  Pattern(gimg.GetLevelCount() * FEATURES),
  GausStructure<GausPatternLevel>(0) {

  for (unsigned k = 0; k < gimg.GetLevelCount(); k++) {
    AddLevel(new GausPatternLevel(gimg.GetLevel(k).GetImage(), r, c));
    r = (unsigned) (r * GAUSSIAN_COMPRS);
    c = (unsigned) (c * GAUSSIAN_COMPRS);
  }
}

GausPattern::~GausPattern() {
}

float GausPattern::operator[](unsigned i) const {
  return GetResponse(i);
}

float &GausPattern::operator[](unsigned i) {
  return GetLevel(i / FEATURES)[i % FEATURES];
}

float GausPattern::GetResponse(unsigned i) const {
  return GetLevel(i / FEATURES)[i % FEATURES];
}

void GausPattern::SetResponse(unsigned i, float resp) {
  GetLevel(i / FEATURES).SetResponse(i % FEATURES, resp);
}

GausPattern *GausPattern::operator-(const Pattern &right) const {
  GausPattern newpat(dims);

  assert(dims == right.GetDimensions());

  for (unsigned i = 0; i < dims; i++)
    newpat.SetResponse(i, GetResponse(i) - right.GetResponse(i));

  return new GausPattern(newpat);
}

GausPattern *GausPattern::operator+(const Pattern &right) const {
  GausPattern newpat(dims);

  assert(dims == right.GetDimensions());

  for (unsigned i = 0; i < dims; i++)
    newpat.SetResponse(i, GetResponse(i) + right.GetResponse(i));

  return new GausPattern(newpat);
}

GausPattern *GausPattern::operator*(float right) const {
  GausPattern newpat(dims);

  for (unsigned i = 0; i < dims; i++)
    newpat.SetResponse(i, GetResponse(i) * right);

  return new GausPattern(newpat);
}

GausPattern *GausPattern::operator/(float right) const {
  GausPattern newpat(dims);

  for (unsigned i = 0; i < dims; i++)
    newpat.SetResponse(i, GetResponse(i) / right);

  return new GausPattern(newpat);
}

const GausPattern &GausPattern::operator-=(const Pattern &right) {
  assert(dims == right.GetDimensions());

  for (unsigned i = 0; i < dims; i++)
    SetResponse(i, GetResponse(i) - right.GetResponse(i));

  return *this;
}

const GausPattern &GausPattern::operator+=(const Pattern &right) {
  assert(dims == right.GetDimensions());

  for (unsigned i = 0; i < dims; i++)
    SetResponse(i, GetResponse(i) + right.GetResponse(i));

  return *this;
}

const GausPattern &GausPattern::operator*=(float right) {
  for (unsigned i = 0; i < dims; i++)
    SetResponse(i, GetResponse(i) * right);

  return *this;
}

const GausPattern &GausPattern::operator/=(float right) {
  for (unsigned i = 0; i < dims; i++)
    SetResponse(i, GetResponse(i) / right);

  return *this;
}

Image *GausPattern::InferImage() const {
  unsigned r = 0, c = 0;

  // Initialize top level
  Image *previmg = new Image(1, 1);
  previmg->SetPixel(0, 0, GetLevel(levcount - 1).GetResponse(PR_GAUS));

  // Work backwards
  for (int k = levcount - 1; k >= 0; k--) {
    Image *img = previmg->Expand();
    GausPatternLevel &lev = GetLevel(k);

    double avg = previmg->GetPixel(r, c);

    // Calculation for center and adjacent pixels is derived in inferimage.mws
    img->SetPixel(r, c, avg - 2.0 * lev.GetResponse(PR_HBAR)
		  - 2.0 * lev.GetResponse(PR_VBAR));
    img->SetPixel(r - 1, c, -avg + 2.0 * lev.GetResponse(PR_HEDG)
		  + 2.0 * lev.GetResponse(PR_HBAR)
		  + 2.0 * lev.GetResponse(PR_GAUS));
    img->SetPixel(r + 1, c, -avg - 2.0 * lev.GetResponse(PR_HEDG)
		  + 2.0 * lev.GetResponse(PR_HBAR)
		  + 2.0 * lev.GetResponse(PR_GAUS));
    img->SetPixel(r, c - 1, -avg + 2.0 * lev.GetResponse(PR_VEDG)
		  + 2.0 * lev.GetResponse(PR_VBAR)
		  + 2.0 * lev.GetResponse(PR_GAUS));
    img->SetPixel(r, c + 1, -avg - 2.0 * lev.GetResponse(PR_VEDG)
		  + 2.0 * lev.GetResponse(PR_VBAR)
		  + 2.0 * lev.GetResponse(PR_GAUS));

    // Reposition r and c
    r = (unsigned) (r / GAUSSIAN_COMPRS) + 1;
    c = (unsigned) (c / GAUSSIAN_COMPRS) + 1;

    delete previmg;
    previmg = img;  // save previous image
  }

  return previmg;
}

void GausPattern::WritePattern(FILE *fp) const {
  for (unsigned k = 0; k < levcount; k++)
    GetLevel(k).WriteResponses(fp);
}

void GausPattern::WritePatternType(FILE *fp) const {
  fprintf(fp, "%s\n", GAUS_PATTERN_TYPE);

  // global data: gaussian levels, dimensions
  fputc(HF_ROLE_GLEVC, fp);
  fputc(HF_TYPE_UNINT, fp);
  fputc(1, fp);

  fputc(HF_ROLE_PDIMC, fp);
  fputc(HF_TYPE_UNINT, fp);
  fputc(1, fp);

  // pattern data: responses
  for (unsigned k = 0; k < levcount; k++) {
    fputc(HF_ROLE_PRESP, fp);
    fputc(HF_TYPE_FLOAT, fp);
    fputc(FEATURES, fp);
  }

  fputc(0, fp);
}

void GausPattern::WritePatternGlobals(FILE *fp) const {
  fwrite(&levcount, sizeof(unsigned), 1, fp);       // levels
  fwrite(&dims, sizeof(unsigned), 1, fp);           // dimensions
}

void GausPattern::PrintPattern() const {
  for (unsigned k = 0; k < levcount; k++)
    GetLevel(k).PrintResponses();
}

// Reads type and globals
GausPattern *GausPattern::ReadPatternType(FILE *fp) {
  unsigned levcount, levcount2, dims;
  int last;

  assert(fgetc(fp) == HF_ROLE_GLEVC);
  assert(fgetc(fp) == HF_TYPE_UNINT);
  assert(fgetc(fp) == 1);

  assert(fgetc(fp) == HF_ROLE_PDIMC);
  assert(fgetc(fp) == HF_TYPE_UNINT);
  assert(fgetc(fp) == 1);

  levcount = 0;
  while (last = fgetc(fp)) {
    levcount++;
    assert(last == HF_ROLE_PRESP);
    assert(fgetc(fp) == HF_TYPE_FLOAT);
    assert(fgetc(fp) == FEATURES);
  }

  // type checks!

  // read in actual global data
  fread(&levcount2, sizeof(unsigned), 1, fp);
  assert(levcount2 == levcount);
  fread(&dims, sizeof(unsigned), 1, fp);
  assert(levcount * FEATURES == dims);

  return new GausPattern(dims);
}

Pattern &GausPattern::ReadPattern(FILE *fp) {
  for (unsigned k = 0; k < levcount; k++)
    GetLevel(k).ReadResponses(fp);

  return *this;
}

Pattern *GausPattern::DuplicateType() const {
  return new GausPattern(dims);
}

Pattern *GausPattern::Duplicate() const {
  return new GausPattern(*this);
}

ePatternType GausPattern::GetPatternType() const {
  return GausPatternType;
}

/***** GausPatternLevel Implementation **************************************/

GausPatternLevel::GausPatternLevel()
  : GausStructureLevel<GausPatternLevel>() {
  for (unsigned i = 0; i < FEATURES; i++)
    response[i] = 0;
}

// Copy constructor
// Note: uses copy's above
GausPatternLevel::GausPatternLevel(const GausPatternLevel &copy)
  : GausStructureLevel<GausPatternLevel>(copy.GetAbove()) {
  for (unsigned i = 0; i < FEATURES; i++)
    response[i] = copy.response[i];
}

// TODO: Change to use unspecified feature functions
GausPatternLevel::GausPatternLevel(const GausImageLevel &glevel,
				   unsigned r, unsigned c) {
  response[PR_GAUS] = glevel.GetImage().CalcFeatureAt(r, c, gaus);
  response[PR_VBAR] = glevel.GetImage().CalcFeatureAt(r, c, vbar);
  response[PR_HBAR] = glevel.GetImage().CalcFeatureAt(r, c, hbar);
  response[PR_VEDG] = glevel.GetImage().CalcFeatureAt(r, c, vedg);
  response[PR_HEDG] = glevel.GetImage().CalcFeatureAt(r, c, hedg);
}

float GausPatternLevel::operator[](unsigned i) const {
  return GetResponse(i);
}

float &GausPatternLevel::operator[](unsigned i) {
  assert(i < FEATURES);
  return response[i];
}

float GausPatternLevel::GetResponse(unsigned i) const {
  assert(i < FEATURES);
  return response[i];
}

void GausPatternLevel::SetResponse(unsigned i, float resp) {
  assert(i < FEATURES);
  response[i] = resp;
}

void GausPatternLevel::WriteResponses(FILE *fp) const {
  fwrite(response, sizeof(float), FEATURES, fp);
}

void GausPatternLevel::ReadResponses(FILE *fp) {
  fread(response, sizeof(float), FEATURES, fp);
}

void GausPatternLevel::PrintResponses() const {
  printf("[");

  for (unsigned i = 0; i < FEATURES - 1; i++)
    printf("%f, ", response[i]);
  printf("%f] ");
}

/***** LocPattern Implementation ********************************************/

LocPattern::LocPattern(Pattern &pat, unsigned r, unsigned c) :
  Pattern(pat.GetDimensions() + 2) {
  basepat = pat.Duplicate();
  locrow = r;
  loccol = c;
}

LocPattern::LocPattern(Pattern *pat, unsigned r, unsigned c) :
  Pattern(pat->GetDimensions() + 2) {
  basepat = pat;
  locrow = r;
  loccol = c;
}

LocPattern::LocPattern(LocPattern &pat) :
  Pattern(pat.dims) {
  basepat = pat.basepat->Duplicate();
  locrow = pat.locrow;
  loccol = pat.loccol;
}

LocPattern::LocPattern(LocPattern *pat) :
  Pattern(pat->dims) {
  basepat = pat->basepat;
  locrow = pat->locrow;
  loccol = pat->loccol;
}

LocPattern::~LocPattern() {
  delete basepat;
}

float LocPattern::operator[](unsigned i) const {
  if (i == 0)
    return locrow;
  if (i == 1)
    return loccol;

  return (*basepat)[i - 2];
}

float &LocPattern::operator[](unsigned i) {
  if (i == 0)
    return *(new float(locrow));
  if (i == 1)
    return *(new float(loccol));

  return (*basepat)[i - 2];
}

float LocPattern::GetResponse(unsigned i) const {
  if (i == 0)
    return (float) locrow;
  if (i == 1)
    return (float) loccol;

  return basepat->GetResponse(i - 2);
}

void LocPattern::SetResponse(unsigned i, float resp) {
  if (i == 0)
    locrow = (unsigned) resp;
  if (i == 1)
    loccol = (unsigned) resp;
  if (i > 1)
    basepat->SetResponse(i - 2, resp);
}

Pattern *LocPattern::operator-(const Pattern &right) const {
  const LocPattern *locright = dynamic_cast<const LocPattern *>(&right);
  assert(locright);

  return new LocPattern(*basepat - *locright->basepat,
			locrow - locright->locrow,
			loccol - locright->loccol);
}

Pattern *LocPattern::operator+(const Pattern &right) const {
  const LocPattern *locright = dynamic_cast<const LocPattern *>(&right);
  assert(locright);

  return new LocPattern(*basepat + *locright->basepat,
			locrow + locright->locrow,
			loccol + locright->loccol);
}

Pattern *LocPattern::operator*(float right) const {
  return new LocPattern(*basepat * right, locrow * right, loccol * right);
}

Pattern *LocPattern::operator/(float right) const {
  return new LocPattern(*basepat / right, locrow / right, loccol / right);
}

const Pattern &LocPattern::operator-=(const Pattern &right) {
  const LocPattern *locright = dynamic_cast<const LocPattern *>(&right);
  assert(locright);

  locrow -= locright->locrow;
  loccol -= locright->loccol;
  *basepat -= *locright->basepat;

  return *this;
}

const Pattern &LocPattern::operator+=(const Pattern &right) {
  const LocPattern *locright = dynamic_cast<const LocPattern *>(&right);
  assert(locright);

  locrow += locright->locrow;
  loccol += locright->loccol;
  *basepat += *locright->basepat;

  return *this;
}

const Pattern &LocPattern::operator*=(float right) {
  locrow *= (unsigned) right;
  loccol *= (unsigned) right;
  *basepat *= right;

  return *this;
}

const Pattern &LocPattern::operator/=(float right) {
  locrow /= (unsigned) right;
  loccol /= (unsigned) right;
  *basepat /= right;
}

void LocPattern::WritePattern(FILE *fp) const {
  basepat->WritePattern(fp);
  fwrite(&locrow, sizeof(unsigned), 1, fp);
  fwrite(&loccol, sizeof(unsigned), 1, fp);
}

void LocPattern::WritePatternType(FILE *fp) const {
  fprintf(fp, "%s\n", LOC_PATTERN_TYPE);
}

// Finish below

void LocPattern::WritePatternGlobals(FILE *fp) const {
}

void LocPattern::PrintPattern() const {
}

LocPattern *LocPattern::ReadPatternType(FILE *fp) {
  return new LocPattern(NULL);
}

Pattern &LocPattern::ReadPattern(FILE *fp) {
  return *(new LocPattern(NULL));
}

// place basepat pattern type within my type definition
Pattern *LocPattern::DuplicateType() const {
  return new LocPattern(basepat->DuplicateType(), 0, 0);
}

Pattern *LocPattern::Duplicate() const {
  return new LocPattern(basepat->Duplicate(), locrow, loccol);
}

ePatternType LocPattern::GetPatternType() const {
  return LocPatternType;
}

Pattern &LocPattern::GetBasePattern() {
  return *basepat;
}

Pattern *LocPattern::SetBasePattern(Pattern *pat) {
  Pattern *oldpat = basepat;
  basepat = pat;
  return oldpat;
}

unsigned LocPattern::GetRow() {
  return locrow;
}

unsigned LocPattern::GetColumn() {
  return loccol;
}

void LocPattern::SetRow(unsigned r) {
  locrow = r;
}

void LocPattern::SetColumn(unsigned c) {
  loccol = c;
}
