/****************************************************************************
 * Pattern -- Individual pattern vector class for Viola's texture algorithm *
 ****************************************************************************
 * "A pattern (or feature vector, observation, or datum) is a single data   *
 * item used by the clustering algorithm. It typically consists of a vector *
 * of d measurements vx = (x1, ..., xd)." -- [TODO: paper biblio]           *
 *                                                                          *
 ****************************************************************************/

#ifndef PATTERN_H
#define PATTERN_H

#define GAUS_PATTERN_TYPE "Gaussian Pattern"
#define LOC_PATTERN_TYPE "Located Pattern"

// classes to be defined
class Pattern;
class GausPattern;
class GausPatternLevel;
class LocPattern;

#include "mystd.h"
#include "gausstruct.h"
#include "gausimage.h"

#define FEATURES        5

enum ePatternType { PatternBaseType,
		    GausPatternType,
		    LocPatternType };

class Pattern {
public:
  Pattern(unsigned d);

  unsigned GetDimensions() const;

  // Pure virtual access functions (response data is pattern-type specific)
  virtual float operator[](unsigned i) const = 0;
  virtual float &operator[](unsigned i) = 0;
  virtual float GetResponse(unsigned i) const = 0;
  virtual void SetResponse(unsigned i, float resp) = 0;

  virtual Pattern *operator-(const Pattern &right) const = 0;
  virtual Pattern *operator+(const Pattern &right) const = 0;
  virtual Pattern *operator*(float right) const = 0;
  virtual Pattern *operator/(float right) const = 0;
  virtual const Pattern &operator-=(const Pattern &right) = 0;
  virtual const Pattern &operator+=(const Pattern &right) = 0;
  virtual const Pattern &operator/=(float right) = 0;
  virtual const Pattern &operator*=(float right) = 0;

  virtual void WritePattern(FILE *fp) const = 0;
  virtual void WritePatternType(FILE *fp) const = 0;
  virtual void WritePatternGlobals(FILE *fp) const = 0;
  virtual void PrintPattern() const = 0;

  static Pattern *SelectPatternType(FILE *fp);
  virtual Pattern &ReadPattern(FILE *fp) = 0;

  virtual Pattern *DuplicateType() const = 0;
  virtual Pattern *Duplicate() const = 0;

  void Zero();        // set all responses to 0

  float SumReduce() const;
  float SumSqrReduce() const;

  Pattern *Sqr() const;  // square each response

  void RandomShift(float factor);

  // Distance measures and tests
  float CartesianDistance(const Pattern &two) const;
  bool NearByResponse(const Pattern &two) const;
  float ResponseDistance(float respone, float resptwo) const;
  bool NearByPattern(const Pattern &two, float Znorm) const;
  float PatternDistance(const Pattern &two) const;

  virtual ePatternType GetPatternType() const;

protected:
  unsigned dims;
};

// A GausPattern is built of smaller pattern vectors, one for each level of
//   a Gaussian Pyramid
// Information on different levels can be combined across patterns if the
//  different levels are separate
class GausPattern : public GausStructure<GausPatternLevel>, public Pattern {
public:
  GausPattern(unsigned i);
  GausPattern(const GausPattern &copy);
  GausPattern(const GausImage &gimg, unsigned r, unsigned c);
  virtual ~GausPattern();

  virtual float operator[](unsigned i) const;
  virtual float &operator[](unsigned i);
  virtual float GetResponse(unsigned i) const;
  virtual void SetResponse(unsigned i, float resp);

  virtual GausPattern *operator-(const Pattern &right) const;
  virtual GausPattern *operator+(const Pattern &right) const;
  virtual GausPattern *operator*(float right) const;
  virtual GausPattern *operator/(float right) const;
  virtual const GausPattern &operator-=(const Pattern &right);
  virtual const GausPattern &operator+=(const Pattern &right);
  virtual const GausPattern &operator/=(float right);
  virtual const GausPattern &operator*=(float right);

  Image *InferImage() const;

  virtual void WritePattern(FILE *fp) const;
  virtual void WritePatternType(FILE *fp) const;
  virtual void WritePatternGlobals(FILE *fp) const;
  virtual void PrintPattern() const;

  static GausPattern *ReadPatternType(FILE *fp);
  virtual Pattern &ReadPattern(FILE *fp);

  virtual Pattern *DuplicateType() const;
  virtual Pattern *Duplicate() const;

  virtual ePatternType GetPatternType() const;

private:
};

// The data for a patter at each level is the five feature functions
class GausPatternLevel : public GausStructureLevel<GausPatternLevel> {
public:
  GausPatternLevel();
  GausPatternLevel(const GausPatternLevel &copy);
  GausPatternLevel(const GausImageLevel &glevel, unsigned r, unsigned c);

  float operator[](unsigned i) const;
  float &operator[](unsigned i);
  float GetResponse(unsigned i) const;
  void SetResponse(unsigned i, float resp);

  void WriteResponses(FILE *fp) const;
  void ReadResponses(FILE *fp);
  void PrintResponses() const;

private:
  float response[FEATURES];
};

class LocPattern : public Pattern {
public:
  LocPattern(Pattern &pat, unsigned r, unsigned c);
  LocPattern(Pattern *pat, unsigned r, unsigned c);
  LocPattern(LocPattern &pat);
  LocPattern(LocPattern *pat);
  virtual ~LocPattern();

  virtual float operator[](unsigned i) const;
  virtual float &operator[](unsigned i);
  virtual float GetResponse(unsigned i) const;
  virtual void SetResponse(unsigned i, float resp);

  virtual Pattern *operator-(const Pattern &right) const;
  virtual Pattern *operator+(const Pattern &right) const;
  virtual Pattern *operator*(float right) const;
  virtual Pattern *operator/(float right) const;
  virtual const Pattern &operator-=(const Pattern &right);
  virtual const Pattern &operator+=(const Pattern &right);
  virtual const Pattern &operator/=(float right);
  virtual const Pattern &operator*=(float right);

  virtual void WritePattern(FILE *fp) const;
  virtual void WritePatternType(FILE *fp) const;
  virtual void WritePatternGlobals(FILE *fp) const;
  virtual void PrintPattern() const;

  static LocPattern *ReadPatternType(FILE *fp);
  virtual Pattern &ReadPattern(FILE *fp);

  virtual Pattern *DuplicateType() const;
  virtual Pattern *Duplicate() const;

  Pattern &GetBasePattern();
  Pattern *SetBasePattern(Pattern *pat);  // return previous basepat

  unsigned GetRow();
  unsigned GetColumn();
  void SetRow(unsigned r);
  void SetColumn(unsigned c);

  virtual ePatternType GetPatternType() const;

private:
  Pattern *basepat;
  unsigned locrow;
  unsigned loccol;
};

#endif
