/****************************************************************************
 * System -- All the information for a complete model comparison            *
 ****************************************************************************
 *                                                                          *
 ****************************************************************************/

#include "system.h"

#define PROB_INCLASS    .5

System::System() {
}

System::System(Model *in, Model *out) {
  models.AddHead(in);
  models.AddTail(out);
}

System::~System() {
  POSITION pos = models.GetHeadPosition();
  while (pos)
    delete models.GetNext(pos);
}

Model &System::GetStandardModel() {
  if (models.GetCount() < 1)
    models.AddHead(new Model());
  return *models.GetHead();
}

Model &System::GetContrastModel() {
  if (models.GetCount() < 1)
    models.AddHead(new Model());
  if (models.GetCount() < 2)
    models.AddTail(new Model());
  return *models.GetTail();
}

unsigned System::GetModelCount() {
  return models.GetCount();
}

void System::SetModelCount(unsigned count) {
  while (models.GetCount() < count)
    models.AddTail(new Model());
}

Model &System::GetModel(unsigned id) {
  POSITION pos = models.GetHead();
  while (id--)
    models.GetNext(pos);
  return *models.GetAt(pos);
}

void System::AddModel(Model *newmodel) {
  models.AddTail(newmodel);
}

void System::AddModel() {
  models.AddTail(new Model());
}

// Probability by comparing the histogram of the test image relative to the
//   model histogram to the histogram of the test image relative to the
//   out-of-class histogram
double System::EntropyProb(const CArray<Pattern *, Pattern *> &pats) {
  List<double, double> results;

  for (unsigned m = 1; m < GetModelCount(); m++)
    results.AddTail(1.0 - BayesRule(models.GetHead().CalcEntropy(pats),
				    models.GetModel(m).CalcEntropy(pats)));

  return results;
}

// Same a EntropyProb, but uses ChiSqr
double System::ChiSqrProb(const CArray<Pattern *, Pattern *> &pats) {
  List<double, double> results;

  for (unsigned m = 1; m < GetModelCount(); m++)
    results.AddTail(1.0 - BayesRule(models.GetHead().CalcChiSqr(pats),
				    model.GetModel(m).CalcChiSqr(pats)));

  return results;
}

// Probability based on checking each pattern against each histogram 
double Model::PatternProb(const CArray<Pattern *, Pattern *> &pats) {
  List<double, double> results;

  for (unsigned m = 1; m < GetModelCount(); m++)
    results.AddTail(BayesRule(models.GetHead().CalcPattern(pats),
			      models.GetModel(m).CalcPattern(pats)));

  return results;
}

// Just calculate Bayes rule:
//  P(C|v) = P(v|C) P(C) / (P(v|C) P(C) + P(v|D) P(D))
//  where P(v|C) is the probability of v in the in-class model C (modelprob)
//  where P(v|D) is the probability of v in the out-of-class model D (antiprob)
//  and P(C) + P(D) = 1, but are otherwise arbitrary (from PROB_INCLASS)
double Model::BayesRule(double modelprob, double antiprob) const {
  if (modelprob || antiprob)
    return modelprob * PROB_INCLASS /
      (modelprob * PROB_INCLASS + antiprob * (1.0 - PROB_INCLASS));
  else
    return .5;
}
