//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: Array.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 10-13-97
// Last modified: 10-23-97
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This class template encapsulates an 1-dimensional array (vector) that can
// dynamically grow and shrink.
//
// TYPE       Type of object stored in the array.
// ARG_TYPE   Type used to reference objects stored in the array.
//            Can be a reference.

#ifndef __ARRAY_H__
#define __ARRAY_H__

#include <Buffer.h>
#include <Object.h>
#include <Archive.h>

template< class TYPE, class ARG_TYPE >
class CArray : public CObject
{
public:
// Construction
  CArray( int nGrowBy = 8 );
  CArray( const CArray& rOther ); 
  CArray& operator=( const CArray& rOther ); 

// Attributes
  BOOL IsEmpty( ) const;
  int GetSize( ) const;
  int GetUpperBound( ) const;
  void SetSize( int nNewSize, int nGrowBy = -1 );

// Operations
  void FreeExtra( );
  void RemoveAll( );
  
// Element Access
  TYPE GetAt( int nIndex ) const;
  void SetAt( int nIndex, ARG_TYPE newElement );
  TYPE& ElementAt( int nIndex );
  const TYPE* GetData( ) const;
  TYPE* GetData( );

// Growing the array
  void SetAtGrow( int nIndex, ARG_TYPE newElement );
  int Add( ARG_TYPE newElement );
  int Append( const CArray& src );
  void Copy( const CArray& src );

// Insertion/Removal
  void InsertAt( int nIndex, ARG_TYPE newElement, int nCount = 1 );
  void InsertAt( int nStartIndex, CArray* pNewArray );
  void RemoveAt( int nIndex, int nCount = 1 );

// Operators
  TYPE operator[]( int nIndex ) const;
  TYPE& operator[]( int nIndex );

  virtual int ClassID( ) const;
  virtual void Serialize( CArchive& ar );

private:
  enum { CLSID = -101 };

  CBuffer< TYPE > m_data;   // the actual data
  int m_nSize;     // # of elements (upperBound - 1)
  int m_nGrowBy;   // grow amount
};



// Construction

template< class TYPE, class ARG_TYPE >
inline CArray< TYPE, ARG_TYPE >::CArray( int nGrowBy ) :
  m_data( nGrowBy )
{
  m_nGrowBy = nGrowBy;
  m_nSize = 0;
}

template< class TYPE, class ARG_TYPE >
inline CArray< TYPE, ARG_TYPE >::CArray( const CArray< TYPE, ARG_TYPE >& rOther ) :
  m_data( rOther.m_data )
{
  m_nGrowBy = rOther.m_nGrowBy;
  m_nSize = rOther.m_nSize;
}

template< class TYPE, class ARG_TYPE >
inline CArray< TYPE, ARG_TYPE >& CArray< TYPE, ARG_TYPE >::operator=( const CArray< TYPE, ARG_TYPE >&
  rOther )
{
  m_data = rOther.m_data;
  m_nSize = rOther.m_nSize;
  m_nGrowBy = rOther.m_nGrowBy;

  return *this;
}



// Attributes

template< class TYPE, class ARG_TYPE >
inline BOOL CArray< TYPE, ARG_TYPE >::IsEmpty( ) const
{
  return m_nSize == 0;
}

template< class TYPE, class ARG_TYPE >
inline int CArray< TYPE, ARG_TYPE >::GetSize( ) const
{
  return m_nSize;
}

template< class TYPE, class ARG_TYPE >
inline int CArray< TYPE, ARG_TYPE >::GetUpperBound( ) const
{
  return m_nSize - 1;
}

template< class TYPE, class ARG_TYPE >
void CArray< TYPE, ARG_TYPE >::SetSize( int nNewSize, int nGrowBy )
{
  m_data.ForceResize( nNewSize );
  //m_nSize = nNewSize;

  if ( nGrowBy > 0 )
    m_nGrowBy = nGrowBy;
}



// Operations

template< class TYPE, class ARG_TYPE >
inline void CArray< TYPE, ARG_TYPE >::FreeExtra( )
{
  m_data.ForceResize( m_nSize );
}

template< class TYPE, class ARG_TYPE >
inline void CArray< TYPE, ARG_TYPE >::RemoveAll( )
{
  m_data.ForceResize( 0 );
  m_nSize = 0;
}



// Element Access

template< class TYPE, class ARG_TYPE >
inline TYPE CArray< TYPE, ARG_TYPE >::GetAt( int nIndex ) const
{
  ASSERT( nIndex >= 0 && nIndex < m_nSize );

  return m_data[ nIndex ];
}

template< class TYPE, class ARG_TYPE >
inline void CArray< TYPE, ARG_TYPE >::SetAt( int nIndex, ARG_TYPE newElement )
{
  ASSERT( nIndex >= 0 && nIndex < m_nSize );

  m_data[ nIndex ] = newElement;
}

template< class TYPE, class ARG_TYPE >
inline TYPE& CArray< TYPE, ARG_TYPE >::ElementAt( int nIndex )
{
  return ( ( TYPE* ) m_data )[ nIndex ];
}

template< class TYPE, class ARG_TYPE >
inline const TYPE* CArray< TYPE, ARG_TYPE >::GetData( ) const
{
  return ( const TYPE* ) m_data;
}

template< class TYPE, class ARG_TYPE >
inline TYPE* CArray< TYPE, ARG_TYPE >::GetData( )
{
  return ( TYPE* ) m_data;
}


// Growing the array

template< class TYPE, class ARG_TYPE >
void CArray< TYPE, ARG_TYPE >::SetAtGrow( int nIndex, ARG_TYPE newElement )
{
  ASSERT( nIndex >= 0 );

  if ( nIndex > m_data.Size( ) - 1 )
    m_data.Resize( nIndex + m_nGrowBy );
  if ( nIndex > m_nSize - 1 )
    m_nSize = nIndex + 1;

  m_data[ nIndex ] = newElement;
}

template< class TYPE, class ARG_TYPE >
int CArray< TYPE, ARG_TYPE >::Add( ARG_TYPE newElement )
{
  if ( m_nSize == m_data.Size( ) )
    m_data.Resize( m_nSize + m_nGrowBy );
  m_data[ m_nSize ] = newElement;

  return m_nSize++;
}

template< class TYPE, class ARG_TYPE >
int CArray< TYPE, ARG_TYPE >::Append( const CArray< TYPE, ARG_TYPE >& src )
{
  if ( m_nSize + src.m_nSize > m_data.Size( ) )
    m_data.Resize( m_nSize + src.m_nSize + m_nGrowBy );
  
  CopyElements( ( TYPE* ) m_data, ( const TYPE* ) src.m_data, src.m_nSize );
  m_nSize += src.m_nSize;

  return m_nSize;
}

template< class TYPE, class ARG_TYPE >
void CArray< TYPE, ARG_TYPE >::Copy( const CArray< TYPE, ARG_TYPE >& src )
{
  m_nSize = src.m_nSize;
  m_data = src.m_data;
  
  return m_nSize;
}


// Insertion / Removal

template< class TYPE, class ARG_TYPE >
void CArray< TYPE, ARG_TYPE >::InsertAt( int nIndex, ARG_TYPE newElement, int nCount  )
{
  ASSERT( nIndex >= 0 && nCount >= 0 );

  if ( m_nSize + nCount > m_data.Size( ) )
    m_data.Resize( m_nSize + nCount + m_nGrowBy );

  if ( nIndex < m_nSize )
  {
    // Destroy elements before overwriting them
    DestructElements( ( TYPE* ) m_data + m_nSize, nCount );

    // We can make shallow copies, because the old elements will be overwritten.
    // The memory blocks may overlap so use memmove!
    memmove( ( TYPE* ) m_data + nIndex + nCount, ( TYPE* ) m_data + nIndex, ( m_nSize - nIndex ) * sizeof( TYPE ) );  

    // Copyconstruct the new elements
    CopyConstructElements( ( TYPE* ) m_data + nIndex, newElement, nCount );
  }
  else
    CopyElements( ( TYPE* ) m_data, newElement, nCount );
  
  m_nSize += nCount;
}

template< class TYPE, class ARG_TYPE >
void CArray< TYPE, ARG_TYPE >::InsertAt( int nStartIndex, CArray* pNewArray )
{
  ASSERT( this != pNewArray );   // No self insertion
  ASSERT( nStartIndex >= 0 );

  if ( m_nSize + pNewArray->m_nSize > m_data.Size( ) )
    m_data.Resize( m_nSize + pNewArray->m_nSize + m_nGrowBy );

  if ( nStartIndex < m_nSize )
  {
    // Destroy elements before overwriting them
    DestructElements( ( TYPE* ) m_data + m_nSize, pNewArray->m_nSize );

    // We can make shallow copies, because the old elements will be overwritten.
    // The memory blocks may overlap so use memmove!
    memmove( ( TYPE* ) m_data + nStartIndex + pNewArray->m_nSize, ( TYPE* ) m_data + nStartIndex, ( m_nSize - nStartIndex ) * sizeof( TYPE ) );  

    // Copyconstruct the new elements
    CopyConstructElements( ( TYPE* ) m_data + nStartIndex, ( TYPE* ) pNewArray->m_data, pNewArray->m_nSize );
  }
  else
    CopyElements( ( TYPE* ) m_data + nStartIndex, ( TYPE* ) pNewArray->m_data, pNewArray->m_nSize );

  m_nSize += pNewArray->m_nSize;
}

template< class TYPE, class ARG_TYPE >
void CArray< TYPE, ARG_TYPE >::RemoveAt( int nIndex, int nCount )
{
  ASSERT( nIndex >= 0 && nIndex < m_nSize );
  ASSERT( nCount > 0 );

  DestructElements( ( TYPE* ) m_data + nIndex, nCount );

  // We can make shallow copies, because the old elements will no longer be used.
  // The memory blocks may overlap so use memmove!
  memmove( ( TYPE* ) m_data + nIndex, ( TYPE* ) m_data + nIndex + nCount, ( m_nSize - nIndex - 1 ) * sizeof( TYPE ) );

  // Reinit unused elements
  ConstructElements( ( TYPE* ) m_data + m_nSize - nCount, nCount );

  m_nSize--;
}



// Operators

template< class TYPE, class ARG_TYPE >
inline TYPE CArray< TYPE, ARG_TYPE >::operator[]( int nIndex ) const
{
  ASSERT( nIndex >= 0 && nIndex < m_nSize );

  return m_data[ nIndex ];
}

template< class TYPE, class ARG_TYPE >
inline TYPE& CArray< TYPE, ARG_TYPE >::operator[]( int nIndex )
{
  ASSERT( nIndex >= 0 && nIndex < m_nSize );

  return m_data[ nIndex ];
}

template< class TYPE, class ARG_TYPE >
int CArray< TYPE, ARG_TYPE >::ClassID( ) const
{
  return CLSID;
}

template< class TYPE, class ARG_TYPE >
void CArray< TYPE, ARG_TYPE >::Serialize( CArchive& ar )
{
  if ( ar.IsStoring( ) )
  {
    ar << m_nSize;
  }
  else
  {
    ar >> m_nSize;
    cerr << "CArray::m_nSize == " << m_nSize << endl;
    SetSize( m_nSize, -1 );
  }
  SerializeElements( ar, ( TYPE* ) m_data, m_nSize );
}

#endif   // __ARRAY_H__


