////////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: BinaryTree.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 11-21-97
// Last modified: 11-26-97
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This class template encapsulates a binary tree.
//
// TYPE       Type of object stored in the binary tree.
// ARG_TYPE   Type used to reference objects stored in the binary tree.
//            Can be a reference.

#ifndef __BINARYTREE_H__
#define __BINARYTREE_H__

#include <TreeBuffer.h>

#define NODE CTreeBuffer< TYPE, 2 >::CTreeBufferNode
#define m_pLeft m_ppRelatives[ 0 ]
#define m_pRight m_ppRelatives[ 1 ]

template< class TYPE, class ARG_TYPE >
class CBinaryTree
{
public:
  // Construction
  CBinaryTree( int nBlockSize = 8 );
   
  // Operations
  void AddRoot( ARG_TYPE newElement );
  void AddLeft( POSITION pos, ARG_TYPE newElement );
  void AddRight( POSITION pos, ARG_TYPE newElement );
  void RemoveAt( POSITION pos );
  void RemoveLeftAt( POSITION pos );
  void RemoveRightAt( POSITION pos );
  
  // Iteration
  POSITION GetRootPosition( ) const;
  TYPE GetNextLeft( POSITION& rPos ) const;
  TYPE GetNextRight( POSITION& rPos ) const;
  void Preorder( POSITION pos, void ( *pfnIteration )
                 ( CBinaryTree*, POSITION, int ), int nLevel = 0 );
  void Inorder( POSITION pos, void ( *pfnIteration )
                ( CBinaryTree*, POSITION, int ), int nLevel = 0 );
  void Postorder( POSITION pos, void ( *pfnIteration )
                  ( CBinaryTree*, POSITION, int ), int nLevel = 0 );
  void ReversedPreorder( POSITION pos, void ( *pfnIteration )
                         ( CBinaryTree*, POSITION, int ), int nLevel = 0 );
  void ReversedInorder( POSITION pos, void ( *pfnIteration )
                        ( CBinaryTree*, POSITION, int ), int nLevel = 0 );
  void ReversedPostorder( POSITION pos, void ( *pfnIteration )
                          ( CBinaryTree*, POSITION, int ), int nLevel = 0 );

  // Retrieval/Modification
  TYPE GetAt( POSITION pos ) const;
  TYPE& GetAt( POSITION pos );
  void SetAt( POSITION pos, ARG_TYPE newElement );
  
  // Status
  int GetDepth( );
  BOOL IsEmpty( ) const;
  
private:
  static void DeleteNode( CBinaryTree*, POSITION pos, int nLevel );
  static void Depth( CBinaryTree*, POSITION pos, int nLevel );

protected:
  int m_nDepth;   // Only used by GetDepth to compute max level
  NODE* m_pRoot;   // Root of the tree
  CTreeBuffer< TYPE, 2 > m_data;   // The nodes of the tree
};



// Construction

template< class TYPE, class ARG_TYPE >
CBinaryTree< TYPE, ARG_TYPE >::CBinaryTree( int nBlockSize ) :
  m_data( nBlockSize )  
{
  m_pRoot = NULL;
}



// Operations

// Add newElement as root (tree must be empty)
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::AddRoot( ARG_TYPE newElement )
{
  ASSERT( IsEmpty( ) );

  m_pRoot = m_data.NewNode( );
  m_pRoot->m_value = newElement;
}

// Add newElement as left son of pos (pos must not have a left son before)
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::
  AddLeft( POSITION pos, ARG_TYPE newElement )
{
  ASSERT( ( ( NODE* ) pos )->m_pLeft == NULL );

  NODE* pNewNode = m_data.NewNode( );
  pNewNode->m_value = newElement;
  ( ( NODE* ) pos )->m_pLeft = pNewNode;
}

// Add newElement as right son of pos (pos must not have a right son before)
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >:: 
  AddRight( POSITION pos, ARG_TYPE newElement )
{
  ASSERT( ( ( NODE* ) pos )->m_pRight == NULL );

  NODE* pNewNode = m_data.NewNode( );
  pNewNode->m_value = newElement;
  ( ( NODE* ) pos )->m_pRight = pNewNode;
}

// Remove both subtrees of pos
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::RemoveAt( POSITION pos )
{
  RemoveLeftAt( pos );
  RemoveRightAt( pos );
}

// Remove left subtree of pos
template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::RemoveLeftAt( POSITION pos )
{
  Postorder( ( ( NODE* ) pos )->m_pLeft, DeleteNode );
  ( ( NODE* ) pos )->m_pLeft = NULL;
}

// Remove right subtree of pos
template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::RemoveRightAt( POSITION pos )
{
  Postorder( ( ( NODE* ) pos )->m_pRight, DeleteNode );
  ( ( NODE* ) pos )->m_pRight = NULL;
}



// Iteration

template< class TYPE, class ARG_TYPE >
inline POSITION CBinaryTree< TYPE, ARG_TYPE >::GetRootPosition( ) const
{
  return m_pRoot;
}

// return the value of pos and move left
template< class TYPE, class ARG_TYPE >
inline TYPE CBinaryTree< TYPE, ARG_TYPE >::GetNextLeft( POSITION& rPos ) const
{
  NODE* pNode = ( NODE* ) rPos;
  rPos = pNode->m_pLeft;
  
  return pNode->m_value;
}

// return the value of pos and move right
template< class TYPE, class ARG_TYPE >
inline TYPE CBinaryTree< TYPE, ARG_TYPE >::GetNextRight( POSITION& rPos ) const
{
  NODE* pNode = ( NODE* ) rPos;
  rPos = pNode->m_pRight;
  
  return pNode->m_value;
}

// make a preorder traverse with pfnIteration
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::Preorder( POSITION pos,
  void ( *pfnIteration )( CBinaryTree*, POSITION, int ), int nLevel )
{
  if ( pos != NULL )
  {
    pfnIteration( this, pos, nLevel );
    Preorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration, nLevel + 1 );
    Preorder( ( ( NODE* ) pos )->m_pRight, pfnIteration, nLevel + 1 );
  }
}

// make an inorder traverse with pfnIteration
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::Inorder( POSITION pos,
  void ( *pfnIteration )( CBinaryTree*, POSITION, int ), int nLevel )
{
  if ( pos != NULL )
  {
    Inorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration, nLevel + 1 );
    pfnIteration( this, pos, nLevel );
    Inorder( ( ( NODE* ) pos )->m_pRight, pfnIteration, nLevel + 1 );
  }
}

// make a postorder traverse with pfnIteration
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::Postorder( POSITION pos,
  void ( *pfnIteration )( CBinaryTree*, POSITION, int ), int nLevel )
{
  if ( pos != NULL )
  {
    Postorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration, nLevel + 1 );
    Postorder( ( ( NODE* ) pos )->m_pRight, pfnIteration, nLevel + 1 );
    pfnIteration( this, pos, nLevel );
  }
}

// make a reversed predorder traverse with pfnIteration
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::ReversedPreorder( POSITION pos,
  void ( *pfnIteration )( CBinaryTree*, POSITION, int ), int nLevel )
{
  if ( pos != NULL )
  {
    pfnIteration( this, pos, nLevel );
    ReversedPreorder( ( ( NODE* ) pos )->m_pRight, pfnIteration, nLevel + 1 );
    ReversedPreorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration, nLevel + 1 );
  }
}

// make a reversed inorder traverse with pfnIteration
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::ReversedInorder( POSITION pos,
  void ( *pfnIteration )( CBinaryTree*, POSITION, int ), int nLevel )
{
  if ( pos != NULL )
  {
    ReversedInorder( ( ( NODE* ) pos )->m_pRight, pfnIteration, nLevel + 1 );
    pfnIteration( this, pos, nLevel );
    ReversedInorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration, nLevel + 1 );
  }
}

// make a reversed postorder traverse with pfnIteration
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::ReversedPostorder( POSITION pos,
  void ( *pfnIteration )( CBinaryTree*, POSITION, int ), int nLevel )
{
  if ( pos != NULL )
  {
    ReversedPostorder( ( ( NODE* ) pos )->m_pRight, pfnIteration, nLevel + 1 );
    ReversedPostorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration, nLevel + 1 );
    pfnIteration( this, pos, nLevel  );
  }
}



// Retrieval/Modification

// Return value of pos
template< class TYPE, class ARG_TYPE >
inline TYPE CBinaryTree< TYPE, ARG_TYPE >::GetAt( POSITION pos ) const
{
  return ( ( NODE* ) pos )->m_value;
}

// Return a reference to the value of pos
template< class TYPE, class ARG_TYPE >
inline TYPE& CBinaryTree< TYPE, ARG_TYPE >::GetAt( POSITION pos )
{
  return ( ( NODE* ) pos )->m_value;
}

// Change the value of pos to newElement
template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::SetAt( POSITION pos, ARG_TYPE newElement )
{
  ( ( NODE* ) pos )->m_value = newElement;
}


  
// Status

// Return the depth of the tree
template< class TYPE, class ARG_TYPE >
CBinaryTree< TYPE, ARG_TYPE >::GetDepth( )
{
  m_nDepth = 0;
  Inorder( m_pRoot, Depth );

  return m_nDepth;
}

// Return TRUE if the tree is empty
template< class TYPE, class ARG_TYPE >
BOOL CBinaryTree< TYPE, ARG_TYPE >::IsEmpty( ) const
{
  return m_pRoot == NULL;
}



// Private member functions

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::
  DeleteNode( CBinaryTree* pBinTree, POSITION pos, int )
{
  pBinTree->m_data.DeleteNode( ( NODE* ) pos );
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::
  Depth( CBinaryTree* pBinTree, POSITION, int nLevel )
{
  if ( nLevel > pBinTree->m_nDepth )
    pBinTree->m_nDepth = nLevel;
}

#undef NODE
#undef m_pLeft
#undef m_pRight

#endif   // __BINARYTREE_H__
