//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: Buffer.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 11-08-97
// Last modified: 11-09-97
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// Low-level class for buffer managing. This class is used to build high-level
// classes like arrays and stacks.
// For non-trivial objects like CString's CBuffer objects are much faster
// than C++ arrays. Imaging the following example:
// You have an array of 1000 CString objects. Now you want to add 10
// additional strings. With the default method of resizing the array you will
// implicitly call 1010 CString constructors to create a new array, 1000
// assignement operators to copy the old array into the new one and 1000
// destructors to destroy the old array.
// CBuffer will call: 10 constructors, 0 assignement operators and 0
// destructors to do the same tasks!!!

#ifndef __BUFFER_H__
#define __BUFFER_H__

#include <Decls.h>
#include <string.h>

template< class TYPE >
class CBuffer
{
public:
// Construction / Destruction
  CBuffer( );
  CBuffer( int nSize );
  CBuffer( const CBuffer& rOther );
  ~CBuffer( );
  
// Operators
  CBuffer& operator=( const CBuffer& rOther );
  TYPE& operator[ ]( int nIndex );
  TYPE operator[ ]( int nIndex ) const;
  operator const TYPE*( ) const; 
  operator TYPE*( ); 

// Attributes
  int Size( ) const;

// Resizing
  void Alloc( int nSize );
  void Realloc( int nSize );
  void Resize( int nSize );
  void ForceResize( int nSize );

private:
  TYPE* m_pData;   // the actual array of data
  int m_nSize;     // # of elements
};



// special new operator that does nothing: needed to call
// constructors
inline void* operator new( unsigned int, void* p )
{
  return p;
}

// call the constructors. For intrinsics this inline function will
// expand to nothing (I checked it).
template< class TYPE >
inline void ConstructElements( TYPE* pElements, int nCount )
{
  ASSERT( nCount >= 0 );

  // call the constructor(s)
  for ( ; nCount--; pElements++ )
    new( ( void* ) pElements ) TYPE;
}

// call the destructors. For intrinsics this inline function will
// expand to nothing (I checked it).
template< class TYPE >
inline void DestructElements( TYPE* pElements, int nCount )
{
  ASSERT( nCount >= 0 );

  // call the destructor(s)
  for ( ; nCount--; pElements++ )
    pElements->~TYPE( );
}

// Copy nCount elements from pSrc to pDest
template< class TYPE >
inline void CopyElements( TYPE* pDest, const TYPE* pSrc, int nCount )
{
  // Default is element-copy using assignment
  while ( nCount-- )
    *pDest++ = *pSrc++;
}

// Copy nCount elements from pSrc to pDest
template< class TYPE >
inline void CopyElements( TYPE* pDest, const TYPE& rSrc, int nCount )
{
  while ( nCount-- )
    *pDest++ = rSrc;
}

template< class TYPE >
inline void CopyConstructElements( TYPE* pDest, const TYPE* pSrc, int nCount )
{
  // Construct elements with copy constructor
  while ( nCount-- )
    new( ( void* ) pDest++ ) TYPE( *pSrc++ );
}

template< class TYPE >
inline void CopyConstructElements( TYPE* pDest, const TYPE& rSrc, int nCount )
{
  // Construct elements with copy constructor
  while ( nCount-- )
    new( ( void* ) pDest++ ) TYPE( rSrc );
}

template< class TYPE >
CBuffer< TYPE >::CBuffer( )
{
  m_nSize = 0;
  m_pData = NULL;
}

template< class TYPE >
inline CBuffer< TYPE >::CBuffer( int nSize )
{
  m_nSize = nSize;
  m_pData = ( TYPE* ) new BYTE[ nSize * sizeof( TYPE ) ];
  
  ConstructElements( m_pData, nSize );
}

template< class TYPE >
inline CBuffer< TYPE >::CBuffer( const CBuffer< TYPE >& rOther )
{
  m_nSize = rOther.m_nSize;
  m_pData = ( TYPE* ) new BYTE[ m_nSize * sizeof( TYPE ) ];
  
  CopyConstructElements( m_pData, rOther.m_pData, rOther.m_nSize );
}

template< class TYPE >
CBuffer< TYPE >::~CBuffer( )
{
  DestructElements( m_pData, m_nSize );

  delete [ ] ( BYTE* ) m_pData;
}

template< class TYPE >
CBuffer< TYPE >& CBuffer< TYPE >::operator=( const CBuffer< TYPE >& rOther )
{
  ASSERT( this != &rOther );   // No self assignement

  Resize( rOther.m_nSize );
  CopyElements( m_pData, rOther.m_pData, rOther.m_nSize );

  return *this;
}

template< class TYPE >
inline int CBuffer< TYPE >::Size( ) const
{
  return m_nSize;
}

template< class TYPE >
inline TYPE& CBuffer< TYPE >::operator[]( int nIndex )
{
  ASSERT( nIndex >= 0 && nIndex <= m_nSize );

  return m_pData[ nIndex ];
}

template< class TYPE >
inline TYPE CBuffer< TYPE >::operator[]( int nIndex ) const
{
  ASSERT( nIndex >= 0 && nIndex <= m_nSize );

  return m_pData[ nIndex ];
}

template< class TYPE >
inline CBuffer< TYPE >::operator const TYPE*( ) const
{
  return m_pData;
}

template< class TYPE >
inline CBuffer< TYPE >::operator TYPE*( )
{
  return m_pData;
}

// Allocate a buffer of nLen elements. Works only
// for buffers constructed with the default constructor.
template< class TYPE >
void CBuffer< TYPE >::Alloc( int nSize )
{
  ASSERT( m_nSize == 0 && m_pData == NULL );

  m_pData = ( TYPE* ) new BYTE[ nSize * sizeof( TYPE ) ];
  ConstructElements( m_pData, nSize );
  
  m_nSize = nSize;
}

// Resize the buffer to at least nLen elements.
// In case of reallocation the buffer content will be lost.
// Realloc is faster than Resize for instrinsics and trivial
// objects but slower for non-trivial objects like CString's.
template< class TYPE >
void CBuffer< TYPE >::Realloc( int nSize )
{
  if ( nSize > m_nSize )
  {
    DestructElements( m_pData, m_nSize );
    delete [ ] ( BYTE* ) m_pData;

    m_pData = ( TYPE* ) new BYTE[ nSize * sizeof( TYPE ) ];
    ConstructElements( m_pData, nSize );
  }
  m_nSize = nSize;
}

// Resize the buffer to at least nSize elements. The buffer
// content will be preserved in any case.
// Resize is slower than Realloc for intrinsics and
// trivial objects but faster for non-trivial objects
// like CString's.
template< class TYPE >
void CBuffer< TYPE >::Resize( int nSize )
{
  if ( nSize > m_nSize )
  {
    TYPE* pNewData = ( TYPE* ) new BYTE[ nSize * sizeof( TYPE ) ];

    // construct new elements
    ConstructElements( pNewData + m_nSize, nSize - m_nSize );
    
    // this will eventually make a shallow copy, but this ok since
    // the old buffer will be deleted
    memcpy( pNewData, m_pData, sizeof( TYPE ) * m_nSize );

    // delete old buffer (no destructors called)
    delete [ ] ( BYTE* ) m_pData;

    m_pData = pNewData;
  }

  m_nSize = nSize;
}

// Same as Resize with the exception that
// the buffer will be exact nSize big.
template< class TYPE >
void CBuffer< TYPE >::ForceResize( int nSize )
{
  if ( nSize > m_nSize )
  {
    TYPE* pNewData = ( TYPE* ) new BYTE[ nSize * sizeof( TYPE ) ];

    // construct new elements
    ConstructElements( pNewData + m_nSize, nSize - m_nSize );
    
    // this will eventually make a shallow copy, but this ok since
    // the old buffer will be deleted
    memcpy( pNewData, m_pData, sizeof( TYPE ) * m_nSize );

    // delete old buffer (no destructors called)
    delete [ ] ( BYTE* ) m_pData;

    m_pData = pNewData;
    m_nSize = nSize;
  }
  else if ( nSize < m_nSize )
  {
    TYPE* pNewData = ( TYPE* ) new BYTE[ nSize * sizeof( TYPE ) ];
    
    // this will eventually make a shallow copy, but this ok since
    // the old buffer will be deleted
    memcpy( pNewData, m_pData, sizeof( TYPE ) * nSize );

    // destruct unnecessary elements
    DestructElements( m_pData + nSize, m_nSize - nSize );
    
    // delete old buffer (no destructors called)
    delete [ ] ( BYTE* ) m_pData;

    m_pData = pNewData;
    m_nSize = nSize;
  }  

}


/*
// Better versions for intrinsics

inline void CopyElements( char* pDest, const char* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( char ) ); }
inline void CopyElements( short* pDest, const short* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( short ) ); }
inline void CopyElements( int* pDest, const int* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( int ) ); }
inline void CopyElements( long* pDest, const long* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( long ) ); }
inline void CopyElements( float* pDest, const float* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( float ) ); }
inline void CopyElements( double* pDest, const double* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( double ) ); }
inline void CopyConstructElements( char* pDest, const char* pSrc, int nCount )
  {  memcpy( pDest, pSrc, nCount * sizeof( char ) ); }
inline void CopyConstructElements( short* pDest, const short* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( short ) ); }
inline void CopyConstructElements( int* pDest, const int* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( int ) ); }
inline void CopyConstructElements( long* pDest, const long* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( long ) ); }
inline void CopyConstructElements( float* pDest, const float* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( float ) ); }
inline void CopyConstructElements( double* pDest, const double* pSrc, int nCount )
  { memcpy( pDest, pSrc, nCount * sizeof( double ) ); }

inline CBuffer< char >& CBuffer< char >::operator=( const CBuffer< char >& rOther )
{
  Realloc( rOther.m_nSize );
  memcpy( m_pData, rOther.m_pData, rOther.m_nSize * sizeof( char ) );

  return *this;
}

inline CBuffer< short >& CBuffer< short >::operator=( const CBuffer< short >& rOther )
{
  Realloc( rOther.m_nSize );
  memcpy( m_pData, rOther.m_pData, rOther.m_nSize * sizeof( char ) );

  return *this;
}

inline CBuffer< int >& CBuffer< int >::operator=( const CBuffer< int >& rOther )
{
  Realloc( rOther.m_nSize );
  memcpy( m_pData, rOther.m_pData, rOther.m_nSize * sizeof( int ) );

  return *this;
}

inline CBuffer< long >& CBuffer< long >::operator=( const CBuffer< long >& rOther )
{
  Realloc( rOther.m_nSize );
  memcpy( m_pData, rOther.m_pData, rOther.m_nSize * sizeof( long ) );

  return *this;
}

inline CBuffer< float >& CBuffer< float >::operator=( const CBuffer< float >& rOther )
{
  Realloc( rOther.m_nSize );
  memcpy( m_pData, rOther.m_pData, rOther.m_nSize * sizeof( float ) );

  return *this;
}

inline CBuffer< double >& CBuffer< double >::operator=( const CBuffer< double >& rOther )
{
  Realloc( rOther.m_nSize );
  memcpy( m_pData, rOther.m_pData, rOther.m_nSize * sizeof( double ) );

  return *this;
}
*/

#endif // __BUFFER_H__
