//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: List.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 10-13-97
// Last modified: 02-11-98
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This class template encapsulates a doubly-linked list of any type.
//
// TYPE       Type of object stored in the list.
// ARG_TYPE   Type used to reference objects stored in the list.
//            Can be a reference.

#ifndef __LIST_H__
#define __LIST_H__

#include <TreeBuffer.h>

#define NODE CTreeBuffer< TYPE, 2 >::CTreeBufferNode
#define m_pPrev m_ppRelatives[ 0 ]
#define m_pNext m_ppRelatives[ 1 ]

template< class TYPE, class ARG_TYPE >
class CList
{
// Operations
public:
  CList( );
  CList( const CList< TYPE, ARG_TYPE >& rOther );
  // ~CList( );

  CList< TYPE, ARG_TYPE >& operator=(
    const CList< TYPE, ARG_TYPE >& rOther );

  int GetCount( ) const;
  BOOL IsEmpty( ) const;
  
  void AddHead( ARG_TYPE item );
  TYPE GetHead( ) const;
  void RemoveHead( );

  void AddTail( ARG_TYPE item );
  void AddTail( CList< TYPE, ARG_TYPE >& items );  // Added by Jimmy
  TYPE GetTail( ) const;
  void RemoveTail( );

  TYPE GetAt( POSITION pos ) const;  // Added by Jimmy
  void SetAt( POSITION pos, ARG_TYPE item );  // Added by Jimmy
  void RemoveAt( POSITION pos );

  POSITION GetHeadPosition( ) const;
  POSITION GetTailPosition( ) const;
  TYPE GetNext( POSITION& pos ) const;
  TYPE GetPrev( POSITION& pos ) const;

  void RemoveAll( );

  BOOL IsValid( ) const;  // Added by Jimmy
  BOOL BelongsToList( POSITION pos ) const;

#ifndef NDEBUG
  void AssertValid( ) const;
#endif

// Helpers
private:
  void Copy( const CList< TYPE, ARG_TYPE >& rOther );

// Attributes
private:
  int m_nSize;
  NODE* m_pHead;
  NODE* m_pTail;
  CTreeBuffer< TYPE, 2 > m_data;   // The nodes of the list
};

/////////////////////////////////////////////////////////////////////////////
// CList< TYPE, ARG_TYPE > inline functions

// Return the number of elements the list contains
template< class TYPE, class ARG_TYPE >
inline int CList< TYPE, ARG_TYPE >::GetCount( ) const 
{
  ASSERT_VALID( this );

  return m_nSize;
}

// Return TRUE if the list is empty, otherwise return FALSE
template< class TYPE, class ARG_TYPE >
inline BOOL CList< TYPE, ARG_TYPE >::IsEmpty( ) const
{
  ASSERT_VALID( this );
  
  return m_nSize == 0;
}

// Return the first element of the list
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetHead( ) const
{
  ASSERT( m_nSize );
  ASSERT_VALID( this );

  return m_pHead->m_value;
}

// Return the last element of the list
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetTail( ) const
{
  ASSERT( m_nSize );
  ASSERT_VALID( this );

  return m_pTail->m_value;
}

// Return the POSITION of the front element
template< class TYPE, class ARG_TYPE >
inline POSITION CList< TYPE, ARG_TYPE >::GetHeadPosition( ) const
{
  ASSERT_VALID( this );

  return m_pHead;
}

// Return the POSITION of the last element
template< class TYPE, class ARG_TYPE >
inline POSITION CList< TYPE, ARG_TYPE >::GetTailPosition( ) const
{
  ASSERT_VALID( this );

  return m_pTail;
}

// Pop all elements
template< class TYPE, class ARG_TYPE >
inline void CList< TYPE, ARG_TYPE >::RemoveAll( )
{
  while ( !IsEmpty( ) )
    RemoveHead( );

  ASSERT( IsEmpty( ) );
}

/////////////////////////////////////////////////////////////////////////////
// CList< TYPE, ARG_TYPE > out-of-line functions

// Default constructor
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >::CList( )
{
  m_nSize = 0;
  m_pHead = NULL;
  m_pTail = NULL;

  ASSERT_VALID( this );
}

// Copy constructor
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >::CList( 
  const CList< TYPE, ARG_TYPE >& rOther )
{
  ASSERT_VALID( &rOther );

  Copy( rOther );
  
  ASSERT_VALID( this );
}

/*// Destructor
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >::~CList( )
{
  ASSERT_VALID( this );
  
  RemoveAll( );
}*/

// Assignment operator
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >& CList< TYPE, ARG_TYPE >::operator=(
  const CList< TYPE, ARG_TYPE >& rOther )
{
  ASSERT_VALID( this );
  ASSERT_VALID( &rOther );
  ASSERT( &rOther != this );
  
  RemoveAll( );
  Copy( rOther );

  ASSERT_VALID( this );

  return *this;
}

// Add 'item' to the front of the list
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::AddHead( ARG_TYPE item )
{
  ASSERT_VALID( this );

  NODE* pNode = m_data.NewNode( );
  pNode->m_value = item;
  pNode->m_pPrev = NULL;
  pNode->m_pNext = m_pHead;
  
  if ( m_pHead )
    m_pHead->m_pPrev = pNode;
  else
    m_pTail = pNode;
  m_pHead = pNode;
    
  m_nSize++;

  ASSERT_VALID( this );
}

// Remove front element
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::RemoveHead( )
{
  ASSERT_VALID( this );
  ASSERT( m_nSize > 0 );

  NODE* pNode = m_pHead;
  m_pHead = m_pHead->m_pNext;
  m_data.DeleteNode( pNode );
  
  if ( m_pHead )
    m_pHead->m_pPrev = NULL;
  else
    m_pTail = NULL;

  m_nSize--;

  ASSERT_VALID( this );
}

// Add 'item' to the back of the list
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::AddTail( ARG_TYPE item )
{
  ASSERT_VALID( this );

  NODE* pNode = m_data.NewNode( );
  pNode->m_value = item;
  pNode->m_pPrev = m_pTail;
  pNode->m_pNext = NULL;
  
  if ( m_pTail )
    m_pTail->m_pNext = pNode;
  else
    m_pHead = pNode;
  m_pTail = pNode;
  
  m_nSize++;

  ASSERT_VALID( this );
}

// Appends one list to another
// Need to copy list nodes (not modify items) or items will be invalid list
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::AddTail( CList< TYPE, ARG_TYPE >& items )
{
  ASSERT_VALID( this );
  ASSERT_VALID( &items );

  for (NODE* pNode = items.m_pHead; pNode; pNode = pNode->m_pNext)
    AddTail(pNode->m_value);

  ASSERT_VALID( this );
}

// Remove back element
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::RemoveTail( )
{
  ASSERT_VALID( this );
  ASSERT( m_nSize > 0 );
  
  NODE* pNode = m_pTail;
  m_pTail = m_pTail->m_pPrev;
  m_data.DeleteNode( pNode );
  
  if ( m_pTail )
    m_pTail->m_pNext = NULL;
  else
    m_pHead = NULL;

  m_nSize--;

  ASSERT_VALID( this );
}

// Return element at pos and move to next position
// pos will be NULL if there are no more elements
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetNext( POSITION& pos ) const
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  NODE* pNode = ( NODE* ) pos;
  pos = pNode->m_pNext;

  return pNode->m_value;
}

// Return element at pos and move to previous position
// pos will be NULL if there are no more elements
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetPrev( POSITION& pos ) const
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  NODE* pNode = ( NODE* ) pos;
  pos = pNode->m_pPrev;

  return pNode->m_value;
}

// Return the element at pos, without changing pos
template< class TYPE, class ARG_TYPE >
TYPE CList< TYPE, ARG_TYPE >::GetAt( POSITION pos ) const
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  NODE* pNode = ( NODE* ) pos;

  return pNode->m_value;
}

// Modify the element at pos, without changing pos
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::SetAt( POSITION pos, ARG_TYPE item )
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  NODE* pNode = ( NODE* ) pos;

  pNode->m_value = item;
}

// Remove element at pos
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::RemoveAt( POSITION pos )
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  NODE* pNode = ( ( NODE* ) pos );

  if ( pNode->m_pPrev )
    pNode->m_pPrev->m_pNext = pNode->m_pNext;
  else
    m_pHead = pNode->m_pNext;

  if ( pNode->m_pNext )
    pNode->m_pNext->m_pPrev = pNode->m_pPrev;
  else
    m_pTail = pNode->m_pPrev;

  m_data.DeleteNode( pNode );
  
  m_nSize--;

  ASSERT_VALID( this );
}


// Private function: copy rOther into this
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::Copy( const CList< TYPE, ARG_TYPE >& rOther )
{
  // Set up empty list
  m_nSize = 0;
  m_pHead = NULL;
  m_pTail = NULL;

  // Copy all elements of rOther
  NODE* pNode = rOther.m_pHead;
  while ( pNode )
  {
    AddTail( pNode->m_value );
    pNode = pNode->m_pNext;
  }
}

// Return TRUE if the list is correctly structured, else return FALSE
template< class TYPE, class ARG_TYPE >
BOOL CList< TYPE, ARG_TYPE >::IsValid( ) const
{
  if ( m_nSize == 0 && ( m_pHead != NULL || m_pTail != NULL ) )
      return FALSE;
  else if ( m_nSize == 1 && ( m_pHead != m_pTail || m_pTail->m_pNext != NULL 
			      || m_pHead->m_pPrev != NULL ) )
    return FALSE;
  else
  {
    int i;
    NODE* pNode;

    pNode = m_pHead;
    for ( i = 0; i < m_nSize - 1; i++ )
      pNode = pNode->m_pNext;
    if ( pNode != m_pTail || ( m_pTail && m_pTail->m_pNext != NULL ) )
      return FALSE;

    pNode = m_pTail;
    for ( i = 0; i < m_nSize - 1; i++ )
      pNode = pNode->m_pPrev;
    if ( pNode != m_pHead || ( m_pHead && m_pHead->m_pPrev != NULL ) )
      return FALSE;
  }

  return TRUE;
}

// Return TRUE if pos is a POSITION of this list else return FALSE
template< class TYPE, class ARG_TYPE >
BOOL CList< TYPE, ARG_TYPE >::BelongsToList( POSITION pos ) const
{
  NODE* pNode = m_pHead;
  while ( pNode )
  {
    if ( pNode == pos )
      return TRUE;
    pNode = pNode->m_pNext;
  }

  return FALSE;
}

// Check class invariant (debug version only)
#ifndef NDEBUG
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::AssertValid( ) const
{
  ASSERT( IsValid( ) );
}
#endif

#endif // __LIST_H__
