#ifndef __SKIPLIST_H__
#define __SKIPLIST_H__

#include <Random.h>
#include "SkipListNode.h"

#define NODE CSkipListNode< KEY, ARG_KEY, VALUE, ARG_VALUE >
typedef void* POSITION;   // Position type for iteration

// Some silly compilers like GNU g++ do not support static
// template members, so we have to use gobal variables
extern int _nSkipLists;
extern CRandom* _pSkipListRandom;

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
class CSkipList
{
public:
  // Construction
  CSkipList( double dP = 0.5 );
  ~CSkipList( );

  // Operations
  BOOL Lookup( ARG_KEY key, VALUE& value ) const;
  void SetAt( ARG_KEY key, ARG_VALUE value );
  VALUE& operator[ ]( ARG_KEY key );
  BOOL RemoveKey( ARG_KEY key );
  void RemoveAll( );
  POSITION GetStartPosition( ) const;
  void GetNextAssoc( POSITION& rNextPosition, KEY& rKey,
                     VALUE& rValue ) const;

  // Status
  int GetCount( ) const;
  BOOL IsEmpty( ) const;

  // Output
  friend ostream& operator<<( ostream& os, const
                    CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >& rList );

private:
  enum { HEADER_GROWBY = 8, OS_WIDTH = 3 };
  
  int RandomLevels( ) const;
  BOOL Search( ARG_KEY key, NODE*& pNode, NODE** ppPrev ) const;
  void ResizeHeader( int nNewSize );
  void PrintNode( ostream& os, NODE* pNodeToPrint ) const;

#ifndef NDEBUG
  void AssertValid( ) const;
  BOOL BelongsToList( POSITION pos ) const;
#endif

private:
  double m_dP;
  int m_nSize;
  int m_nLevels;
  int m_nHeaderSize;
  NODE m_header;
};

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::CSkipList( double dP ) :
  m_nHeaderSize( HEADER_GROWBY ), m_header( m_nHeaderSize )
{
  ASSERT( dP > 0.0 && dP < 1.0 );

  m_dP = dP;
  m_nSize = 0;
  m_nLevels = 1;

  ASSERT_VALID( this );

  // Setup random generator if this is the first list
  if ( _nSkipLists++ == 0 )
  {
    _pSkipListRandom = new CRandom( );
    _pSkipListRandom->Shuffle( );
  }
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::~CSkipList( )
{
  RemoveAll( );

  // Delete random generator if this is the last list
  if ( --_nSkipLists == 0 )
    delete _pSkipListRandom;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
BOOL CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  Lookup( ARG_KEY key, VALUE& rValue ) const
{
  ASSERT_VALID( this );

  NODE* pNode = ( NODE* ) &m_header;
  for ( int i = m_nLevels - 1; i >= 0; i-- )
  {
    while( pNode->m_ppNext[ i ] != NULL &&
           pNode->m_ppNext[ i ]->m_key < key )
      pNode = pNode->m_ppNext[ i ];
  }

  pNode = *pNode->m_ppNext;

  if ( pNode != NULL && pNode->m_key == key )
  {
    rValue = pNode->m_value;
    return TRUE;
  }
  
  return FALSE;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
inline void CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  SetAt( ARG_KEY key, ARG_VALUE value )
{
  ( *this )[ key ] = value;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
BOOL CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  RemoveKey( ARG_KEY key )
{
  ASSERT_VALID( this );

  NODE* pNode;
  NODE** ppPrev = new NODE*[ m_nLevels + 1 ];

  if ( Search( key, pNode, ppPrev ) )
  {
    for ( int i = 0; i < m_nLevels; i++ )
    {
      if ( ppPrev[ i ]->m_ppNext[ i ] == pNode )
        ppPrev[ i ]->m_ppNext[ i ] = pNode->m_ppNext[ i ];
    }

    delete [ ] ppPrev;
    delete pNode;

    // Adjust m_nLevels
    while ( m_nLevels > 1 && m_header.m_ppNext[ m_nLevels - 1 ] == NULL )
      m_nLevels--;

    m_nSize--;

    return TRUE;
  }

  delete [ ] ppPrev;

  return FALSE;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
VALUE& CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  operator[ ]( ARG_KEY key )
{
  ASSERT_VALID( this );

  NODE* pNode;
  NODE** ppPrev = new NODE*[ m_nLevels + 1 ];

  if ( !Search( key, pNode, ppPrev ) )
  {
    int nNewLevels = RandomLevels( );
    if ( nNewLevels > m_nLevels )
    {
      // There must be only one new level
      ASSERT( nNewLevels == m_nLevels + 1 );

      ResizeHeader( nNewLevels );
      ppPrev[ m_nLevels++ ] = &m_header;   
    }
    pNode = new NODE( nNewLevels, key );
    for ( int i = 0; i < nNewLevels; i++ )
    {
      pNode->m_ppNext[ i ] = ppPrev[ i ]->m_ppNext[ i ];
      ppPrev[ i ]->m_ppNext[ i ] = pNode;
    }

    m_nSize++;
  }

  delete [ ] ppPrev;

  return pNode->m_value;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
void CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::RemoveAll( )
{
  ASSERT_VALID( this );

  NODE* pNode = *m_header.m_ppNext;
  NODE* pNext;

  while ( pNode != NULL )
  {
    pNext = *pNode->m_ppNext;
    delete pNode;
    pNode = pNext;
  }

  m_nSize = 0;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
inline POSITION CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  GetStartPosition( ) const
{
  ASSERT_VALID( this );

  return *m_header.m_ppNext;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
void CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  GetNextAssoc( POSITION& rNextPosition, KEY& rKey, VALUE& rValue ) const
{
  ASSERT( BelongsToList( rNextPosition ) );
  ASSERT_VALID( this );
  
  NODE* pNode = ( NODE* ) rNextPosition;
  ASSERT( pNode != NULL );

  rNextPosition = *pNode->m_ppNext;
  rKey = pNode->m_key;
  rValue = pNode->m_value;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
inline int CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  GetCount( ) const
{
  ASSERT_VALID( this );

  return m_nSize;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
inline BOOL CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  IsEmpty( ) const
{
  ASSERT_VALID( this );

  return m_nSize == 0;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
int CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::RandomLevels( ) const
{
  int nNewLevels = 1;
  while ( _pSkipListRandom->Frac( ) < m_dP )
    nNewLevels++;

  // Fix the dice
  return min( nNewLevels, m_nLevels + 1 );
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
BOOL CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  Search( ARG_KEY key, NODE*& pNode, NODE** ppPrev ) const
{
  pNode = ( NODE* ) &m_header;
  for ( int i = m_nLevels - 1; i >= 0; i-- )
  {
    while( pNode->m_ppNext[ i ] != NULL &&
           pNode->m_ppNext[ i ]->m_key < key )
      pNode = pNode->m_ppNext[ i ];

    ppPrev[ i ] = pNode;
  }

  pNode = *pNode->m_ppNext;

  if ( pNode != NULL && pNode->m_key == key )
    return TRUE;

  return FALSE;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
void CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  ResizeHeader( int nNewSize )
{
  if ( nNewSize > m_nHeaderSize )
  {
    NODE** ppOldNext = m_header.m_ppNext;

    m_header.m_ppNext = new NODE*[ m_nHeaderSize + HEADER_GROWBY ];
    memcpy( m_header.m_ppNext, ppOldNext, sizeof( NODE* ) * m_nHeaderSize );
    memset( m_header.m_ppNext + m_nHeaderSize, 0,
            sizeof( NODE* ) * HEADER_GROWBY ); 
    delete [ ] m_header.m_ppNext;

    m_nHeaderSize += HEADER_GROWBY;
  }
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
void CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  PrintNode( ostream& os, NODE* pNodeToPrint ) const
{
  // Determine number of levels of pNodeToPrint
  int nLevels;
  NODE* pNode;
  for ( nLevels = m_nLevels; nLevels > 0; nLevels-- )
  {
    pNode = ( NODE* ) &m_header;
    while( pNode != NULL && pNode != pNodeToPrint )
      pNode = pNode->m_ppNext[ nLevels - 1 ];
    if ( pNode == pNodeToPrint )
      break;
  }

  int i;
  if ( pNodeToPrint == &m_header )
    os << "l: " << nLevels << ". PTR: ";
  else
    os << "k: " << setw( OS_WIDTH ) << pNodeToPrint->m_key << " v: "
       << setw( OS_WIDTH ) << pNodeToPrint->m_value
       << " l: " << nLevels << ". PTR: ";

  for ( i = 0; i < nLevels - 1; i++ )
  {
    if ( pNodeToPrint->m_ppNext[ i ] != NULL )
      os << i << ": " << setw( OS_WIDTH )
         << pNodeToPrint->m_ppNext[ i ]->m_key << ", ";
    else
      os << i << ": " << "END, ";
  }
  if ( pNodeToPrint->m_ppNext[ i ] != NULL )
      os << i << ": " << setw( OS_WIDTH ) 
         << pNodeToPrint->m_ppNext[ i ]->m_key << endl;
    else
      os << i << ": " << "END" << endl;
}

template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
ostream& operator<<( ostream& os, const
                     CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >& rList )
{
  ASSERT_VALID( &rList );

  os << "SkipList probability: " << rList.m_dP << endl;
  os << "HD                  ";
  rList.PrintNode( os, ( NODE* ) &rList.m_header );
  
  NODE* pNode;
  for ( int i = rList.m_nLevels - 1; i >= 0; i-- )
  {
    os << endl;

    pNode = rList.m_header.m_ppNext[ i ];
    while( pNode != NULL )
    {
      os << "i:" << setw( rList.OS_WIDTH ) << i << " ";
      rList.PrintNode( os, pNode );
      pNode = pNode->m_ppNext[ i ];
    }
  }

  return os;
}


// Check class invariant (debug version only)
#ifndef NDEBUG
template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
void CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::AssertValid( ) const
{
  int i;
  if ( m_nSize == 0 )
  {
    for ( i = 0; i < m_nLevels; i++ )
      ASSERT( m_header.m_ppNext[ i ] == NULL );
  }
  else
  {
    NODE* pNode;

    // Level 0 must have exact m_nSize elements
    pNode = *m_header.m_ppNext;
    for ( i = 0; i < m_nSize; i++ )
      pNode = *pNode->m_ppNext;
    ASSERT( pNode == NULL );
    
    int nOldSize = 0;
    for ( int j = m_nLevels - 1; j > 0; j-- )
    {
      pNode = m_header.m_ppNext[ j ];
      for ( i = 0; i < m_nSize && pNode != NULL; i++ )
        pNode = pNode->m_ppNext[ j ];

      ASSERT( pNode == NULL );
      // This level must have not have less elements than the previous level
      ASSERT( i >= nOldSize );
      nOldSize = i;
    }
  }
}

// Return TRUE if pos is a POSITION of this skiplist else return FALSE
// (debug version only)
template< class KEY, class ARG_KEY, class VALUE, class ARG_VALUE >
BOOL CSkipList< KEY, ARG_KEY, VALUE, ARG_VALUE >::
  BelongsToList( POSITION pos ) const
{
  NODE* pNode = *m_header.m_ppNext;
  while ( pNode )
  {
    if ( pNode == pos )
      return TRUE;
    pNode = *pNode->m_ppNext;
  }

  return FALSE;
}

#endif   // NDEBUG

#endif   // __SKIPLIST_H__
