#include <TreeBuffer.h>
#include <iomanip.h>

#define NODE CTreeBufferNode< TYPE, 2 >
#define m_pLeft m_ppRelatives[ 0 ]
#define m_pRight m_ppRelatives[ 1 ]

template< class TYPE, class ARG_TYPE >
class CBinaryTree
{
public:
  CBinaryTree( );

  // Attributes
  int GetDepth( ) const;
  int GetDepth( POSITION pos ) const;
  TYPE GetAt( POSITION pos ) const;
  TYPE& GetAt( POSITION pos );
  
  void AddLeft( ARG_TYPE newElement );
  void AddRight( );
  void RemoveAt( POSITION pos );
  void RemoveLeftAt( POSITION pos );
  void RemoveRightAt( POSITION pos );
  
  // Iteration
  POSITION GetRootPosition( ) const;
  POSITION GetPosition( ARG_TYPE element ) const;
  TYPE GetNextLeft( POSITION& rPos ) const;
  TYPE GetNextRight( POSITION& rPos ) const;
  void Preorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) );
  void Inorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) );
  void Postorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) );
  void ReversedPreorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) );
  void ReversedInorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) );
  void ReversedPostorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) );
  
  // Output
  void Print( );
  
private:
  BOOL Search( ARG_TYPE element, NODE**& pNode ) const;
  static void PrintNode( CBinaryTree*, POSITION pos );
  static void PrintNodeLevel( CBinaryTree*, POSITION pos );
  static void PrintNodeSpecial( CBinaryTree*, POSITION pos );
  static void Delete( CBinaryTree*, POSITION pos );

private:
  int m_nDepth;
  NODE* m_pRoot;
  CTreeBuffer< TYPE, 2 > m_data;
};

template< class TYPE, class ARG_TYPE >
CBinaryTree< TYPE, ARG_TYPE >::CBinaryTree( )
{
  m_nDepth = 0;
  m_pRoot = NULL;
}

template< class TYPE, class ARG_TYPE >
inline CBinaryTree< TYPE, ARG_TYPE >::GetDepth( ) const
{
  return m_nDepth;
}

template< class TYPE, class ARG_TYPE >
inline int CBinaryTree< TYPE, ARG_TYPE >::GetDepth( POSITION pos ) const
{
  int nDepth = 0;
  NODE* pNode = m_pRoot;
  ARG_TYPE element = ( ( NODE* ) pos )->m_value;

  while ( pNode != NULL )
  {
    if ( element < pNode->m_value )
      pNode = pNode-> m_pLeft;
    else if ( element > pNode->m_value )
      pNode = pNode->m_pRight;
    else
      return nDepth;

    nDepth++;
  }

  return -1;
}

template< class TYPE, class ARG_TYPE >
TYPE CBinaryTree< TYPE, ARG_TYPE >::GetAt( POSITION pos ) const
{
  return ( ( NODE* ) pos )->m_value;
}

template< class TYPE, class ARG_TYPE >
TYPE& CBinaryTree< TYPE, ARG_TYPE >::GetAt( POSITION pos )
{
  return ( ( NODE* ) pos )->m_value;
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::Add( ARG_TYPE newElement )
{
  NODE** ppNode;

  if ( !Search( newElement, ppNode ) )
  {
    *ppNode = m_data.NewNode( );
    ( **ppNode ).m_value = newElement;
  }
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::RemoveAt( POSITION pos )
{
  RemoveLeftAt( pos );
  RemoveRightAt( pos );
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::RemoveLeftAt( POSITION pos )
{
  Postorder( ( ( NODE* ) pos )->m_pLeft, Delete );
  ( ( NODE* ) pos )->m_pLeft = NULL;
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::RemoveRightAt( POSITION pos )
{
  Postorder( ( ( NODE* ) pos )->m_pRight, Delete );
  ( ( NODE* ) pos )->m_pRight = NULL;
}

template< class TYPE, class ARG_TYPE >
inline POSITION CBinaryTree< TYPE, ARG_TYPE >::GetRootPosition( ) const
{
  return m_pRoot;
}

template< class TYPE, class ARG_TYPE >
inline POSITION CBinaryTree< TYPE, ARG_TYPE >::GetPosition( ARG_TYPE element ) const
{
  NODE** ppNode;
  Search( element, ppNode );

  return *ppNode;
}

template< class TYPE, class ARG_TYPE >
inline TYPE CBinaryTree< TYPE, ARG_TYPE >::GetNextLeft( POSITION& rPos ) const
{
  NODE* pNode = ( NODE* ) rPos;
  rPos = pNode->m_pLeft;
  
  return pNode->m_value;
}

template< class TYPE, class ARG_TYPE >
inline TYPE CBinaryTree< TYPE, ARG_TYPE >::GetNextRight( POSITION& rPos ) const
{
  NODE* pNode = ( NODE* ) rPos;
  rPos = pNode->m_pLeft;
  
  return pNode->m_value;
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::Preorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) )
{
  if ( pos != NULL )
  {
    pfnIteration( this, pos );
    Preorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration );
    Preorder( ( ( NODE* ) pos )->m_pRight, pfnIteration );
  }
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::Inorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) )
{
  if ( pos != NULL )
  {
    Inorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration );
    pfnIteration( this, pos );
    Inorder( ( ( NODE* ) pos )->m_pRight, pfnIteration );
  }
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::Postorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) )
{
  if ( pos != NULL )
  {
    Postorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration );
    Postorder( ( ( NODE* ) pos )->m_pRight, pfnIteration );
    pfnIteration( this, pos );
  }
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::ReversedPreorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) )
{
  if ( pos != NULL )
  {
    pfnIteration( pos, ( ( NODE* ) pos )->m_value );
    ReversedPreorder( ( ( NODE* ) pos )->m_pRight, pfnIteration );
    ReversedPreorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration );
  }
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::ReversedInorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) )
{
  if ( pos != NULL )
  {
    ReversedInorder( ( ( NODE* ) pos )->m_pRight, pfnIteration );
    pfnIteration( this, pos );
    ReversedInorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration );
  }
}

template< class TYPE, class ARG_TYPE >
inline void CBinaryTree< TYPE, ARG_TYPE >::ReversedPostorder( POSITION pos, void ( *pfnIteration )( CBinaryTree*, POSITION ) )
{
  if ( pos != NULL )
  {
    ReversedPostorder( ( ( NODE* ) pos )->m_pRight, pfnIteration );
    ReversedPostorder( ( ( NODE* ) pos )->m_pLeft, pfnIteration );
    pfnIteration( this, pos );
  }
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::Print( )
{
  cout << "Preorder:" << endl;
  Preorder( m_pRoot, PrintNode );
  cout << endl << "Preorder levels: " << endl;
  Preorder( m_pRoot, PrintNodeLevel );

  cout << endl << endl << "Inorder:" << endl;
  Inorder( m_pRoot, PrintNode );
  cout << endl << "Inorder levels: " << endl;
  Inorder( m_pRoot, PrintNodeLevel );

  cout << endl << endl << "Postorder:" << endl;
  Postorder( m_pRoot, PrintNode );
  cout << endl << "Postorder levels: " << endl;
  Postorder( m_pRoot, PrintNodeLevel );

  cout << endl << endl << "Special:" << endl;
  ReversedInorder( m_pRoot, PrintNodeSpecial );
  cout << endl;
}

template< class TYPE, class ARG_TYPE >
BOOL CBinaryTree< TYPE, ARG_TYPE >::Search( ARG_TYPE element, NODE**& ppNode ) const
{
  ppNode = ( NODE** ) &m_pRoot;
  while ( *ppNode != NULL )
  {
    if ( element < ( **ppNode ).m_value )
      ppNode = &( **ppNode ).m_pLeft;
    else if ( element > ( **ppNode ).m_value )
      ppNode = &( **ppNode ).m_pRight;
    else
      return TRUE;
  }

  return FALSE;
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::PrintNode( CBinaryTree* pBinTree, POSITION pos )
{
  cout << pBinTree->GetAt( pos ) << ' ';
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::PrintNodeLevel( CBinaryTree* pBinTree, POSITION pos )
{
  cout << pBinTree->GetDepth( pos ) << ' ';
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::PrintNodeSpecial( CBinaryTree* pBinTree, POSITION pos )
{
  int nDepth = pBinTree->GetDepth( pos );
  for ( int i = 0; i < nDepth; i++ )
    cout << '\t';
  cout << pBinTree->GetAt( pos ) << endl;
}

template< class TYPE, class ARG_TYPE >
void CBinaryTree< TYPE, ARG_TYPE >::Delete( CBinaryTree* pBintree, POSITION pos )
{
  pBintree->m_data.DeleteNode( ( NODE* ) pos );
}
