//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: Stack.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 11-09-97
// Last modified: 11-09-97
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This class template encapsulates a last-in-first-out (LIFO) stack
// that can dynamically grow and shrink. 
//
// TYPE       Type of object stored in the stack.
// ARG_TYPE   Type used to reference objects stored in the stack.
//            Can be a reference.

#ifndef __STACK_H__
#define __STACK_H__

#include <Buffer.h>

template< class TYPE, class ARG_TYPE >
class CStack
{
public:
// Construction
  CStack( int nGrowBy = 8 );

// Attributes
  int GetSize( ) const;
  BOOL IsEmpty( ) const;
  TYPE Peek( ) const;

// Operations
  TYPE Pop( );
  void Push( ARG_TYPE newElement );
  void FreeExtra( );
  void RemoveAll( );
  
// Implementation
private:
  CBuffer< TYPE > m_data;   // the actual data
  int m_nSize;              // size of the stack
  int m_nGrowBy;            // grow amount
};

template< class TYPE, class ARG_TYPE >
CStack< TYPE, ARG_TYPE >::CStack( int nGrowBy ) :
  m_data( nGrowBy )
{
  m_nSize = 0;
  m_nGrowBy = nGrowBy;
}

template< class TYPE, class ARG_TYPE >
inline int CStack< TYPE, ARG_TYPE >::GetSize( ) const
{
  return m_nSize;
}

template< class TYPE, class ARG_TYPE >
inline BOOL CStack< TYPE, ARG_TYPE >::IsEmpty( ) const
{
  return m_nSize == 0;
}

// Return last pushed element (stack must not be empty)
template< class TYPE, class ARG_TYPE >
inline TYPE CStack< TYPE, ARG_TYPE >::Peek( ) const
{
  ASSERT( m_nSize >= 0 );

  return m_data[ m_nSize - 1 ];
}

template< class TYPE, class ARG_TYPE >
inline TYPE CStack< TYPE, ARG_TYPE >::Pop( )
{
  ASSERT( m_nSize >= 0 );

  return m_data[ --m_nSize ];
}

template< class TYPE, class ARG_TYPE >
inline void CStack< TYPE, ARG_TYPE >::Push( ARG_TYPE newElement )
{
  if ( m_nSize == m_data.Size( ) )
    m_data.Resize( m_nSize + m_nGrowBy );
  
  m_data[ m_nSize++ ] = newElement;
}

// Free any extra memory
template< class TYPE, class ARG_TYPE >
inline void CStack< TYPE, ARG_TYPE >::FreeExtra( )
{
  m_data.ForceResize( m_nSize );
}

// Remove all elements from stack
template< class TYPE, class ARG_TYPE >
inline void CStack< TYPE, ARG_TYPE >::RemoveAll( )
{
  m_data.ForceResize( 0 );
  m_nSize = 0;
}

#endif   // __STACK_H__
