//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: StrClass.cpp
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 10-13-97
// Last modified: 02-11-98
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This file contains the implementation of the string class.

#include <stdarg.h>
#include <Archive.h>
#include <StrClass.h>

// Construction/Destruction

// default constructor: construct an empty string
CString::CString( ) : m_buf( 1 )
{
  m_nLen = 0;
  m_buf[ 0 ] = '\0';
}

// copy constructor (no loonger needed, because of the use of CBuffer)
/*CString::CString( const CString& stringSrc )
{
  NewString( stringSrc.m_nLen );
  memcpy( m_pszString, stringSrc.m_pszString, m_nLen );
}*/

// constructor: create a string of length nRepeat and fill it with ch
CString::CString( char ch, int nRepeat ) : m_buf( nRepeat + 1 )
{
  m_nLen = nRepeat;
  
  //for ( int i = 0; i < nRepeat; i++ )
  //  m_buf[ i ] = ch;
  memset( ( char* ) m_buf, ch, nRepeat );
  m_buf[ nRepeat ] = '\0';
}

// constructor: create a string with the c-string lpch, copy only first
// nLength characters
CString::CString( const char* lpch, int nLength ) : m_buf( nLength + 1 )
{
  m_nLen = nLength;

  memcpy( ( char* ) m_buf, lpch, nLength );
  m_buf[ nLength ] = '\0';
}

// constructor: create a string with the c-string lpsz
CString::CString( const unsigned char* lpsz ) : m_buf( strlen( ( const char* ) lpsz ) + 1 )
{
  m_nLen = m_buf.Size( ) - 1;

  memcpy( ( char* ) m_buf, lpsz, m_nLen );
  m_buf[ m_nLen ] = '\0';
}

// constructor: create a string with the c-string lpsz
CString::CString( const char* lpsz ) : m_buf( strlen( lpsz ) + 1 )
{
  m_nLen = m_buf.Size( ) - 1;

  memcpy( ( char* ) m_buf, lpsz, m_nLen );
  m_buf[ m_nLen ] = '\0';
}

// protected constructor: create an uninitialized string (used by operator+)
CString::CString( int nLen ) : m_buf( nLen + 1 )
{
  m_nLen = nLen;

  m_buf[ nLen ] = '\0';
}



// Assignement/Concatenation

// assignement operator
/*const CString& CString::operator=( const CString& rOther )
{
  m_buf = rOther.m_buf;
  
  return *this;
}*/

const CString& CString::operator=( char ch )
{
  ASSERT_VALID( this );

  m_nLen = 1;

  m_buf.Realloc( 2 );
  m_buf[ 0 ] = ch;
  m_buf[ 1 ] = '\0';
  
  return *this;
}

const CString& CString::operator=( const unsigned char* lpsz )
{
  ASSERT_VALID( this );

  m_nLen = strlen( ( const char* ) lpsz );

  m_buf.Realloc( m_nLen + 1 );
  memcpy( ( char* ) m_buf, lpsz, m_nLen );
  m_buf[ m_nLen ] = '\0';

  return *this;
}

const CString& CString::operator=( const char* lpsz )
{
  ASSERT_VALID( this );

  m_nLen = strlen( lpsz );

  m_buf.Realloc( m_nLen + 1 );
  memcpy( ( char* ) m_buf, lpsz, m_nLen );
  m_buf[ m_nLen ] = '\0';

  return *this;
}

// friend function: concatenate string1 and string2
CString operator+( const CString& string1, const CString& string2 )
{
  ASSERT_VALID( &string1 );
  ASSERT_VALID( &string2 );

  CString result( string1.m_nLen + string2.m_nLen );
  memcpy( ( char* ) result.m_buf, ( const char* ) string1.m_buf, string1.m_nLen );
  memcpy( ( char* ) result.m_buf + string1.m_nLen, ( const char* ) string2.m_buf, string2.m_nLen );

  return result;
}

// friend function: concatenate string and ch
CString operator+( const CString& string, char ch )
{
  ASSERT_VALID( &string );

  CString result( string.m_nLen + 1 );
  memcpy( ( char* ) result.m_buf, ( const char* ) string.m_buf, string.m_nLen );
  result.m_buf[ result.m_nLen - 1 ] = ch;
  
  return result;
}

// friend function: concatenate ch and string
CString operator+( char ch, const CString& string )
{
  ASSERT_VALID( &string );

  CString result( string.m_nLen + 1 );
  result.m_buf[ 0 ] = ch;
  memcpy( ( char* ) result.m_buf + 1, ( const char* ) string.m_buf, string.m_nLen );
  
  return result;
}

// friend function: concatenate string and lpsz
CString operator+( const CString& string, const char* lpsz )
{
  ASSERT_VALID( &string );

  int nLen = strlen( lpsz );
  CString result( string.m_nLen + nLen );
  memcpy( ( char* ) result.m_buf, ( const char* ) string.m_buf, string.m_nLen );
  memcpy( ( char* ) result.m_buf + string.m_nLen, lpsz, nLen );

  return result;
}

// friend function: concatenate lpsz and string
CString operator+( const char* lpsz, const CString& string )
{
  ASSERT_VALID( &string );

  int nLen = strlen( lpsz );
  CString result( nLen + string.m_nLen );
  memcpy( ( char* ) result.m_buf, lpsz, nLen );
  memcpy( ( char* ) result.m_buf + nLen, ( const char* ) string.m_buf, string.m_nLen );

  return result;
}

const CString& CString::operator+=( const CString& string )
{
  ASSERT_VALID( this );
  ASSERT_VALID( &string );

  m_nLen += string.m_nLen;
  
  m_buf.Resize( m_nLen + 1 );
  memcpy( ( char* ) m_buf + m_nLen - string.m_nLen, ( const char* ) string.m_buf, string.m_nLen );
  m_buf[ m_nLen ] = '\0';
  
  return *this;
}

const CString& CString::operator+=( char ch )
{
  ASSERT_VALID( this );

  m_nLen++;
  
  m_buf.Resize( m_nLen + 1 );
  m_buf[ m_nLen - 1 ] = ch;
  m_buf[ m_nLen ] = '\0';

  return *this;
}

const CString& CString::operator+=( const char* lpsz )
{
  ASSERT_VALID( this );

  int nLen = strlen( lpsz );

  m_buf.Resize( m_nLen + nLen + 1 );
  memcpy( m_buf + m_nLen, lpsz, nLen );
  
  m_nLen += nLen;
  m_buf[ m_nLen ] = '\0';
  
  return *this;
}



// Other Conversions

void CString::Format( const char* lpszFormat, ... )
{
  ASSERT_VALID( this );

  // Let's hope the output will be no more than 1024 chars
  m_buf.Realloc( 1025 );
  
  va_list argList;
  va_start( argList, lpszFormat );
  m_nLen = vsprintf( m_buf, lpszFormat, argList );
  va_end( argList );

  ASSERT( m_nLen >= 0 && m_nLen <= 1024 );
  
  FreeExtra( );
}

void CString::TrimLeft( )
{
  ASSERT_VALID( this );

  char* pszFirst = m_buf;

  while ( *pszFirst == ' ' || *pszFirst == '\t' || *pszFirst == '\n' )
    pszFirst++;

  m_nLen -= pszFirst - ( char* ) m_buf;
  // The memory blocks overlap, we have to use memmove
  memmove( ( char* ) m_buf, pszFirst, m_nLen + 1 );
}

void CString::TrimRight( )
{
  ASSERT_VALID( this );

  char* pszLast = ( char* ) m_buf + m_nLen - 1;

  while ( *pszLast == ' ' || *pszLast == '\t' || *pszLast == '\n' )
    pszLast--;

  m_nLen = pszLast - ( char* ) m_buf + 1;
  m_buf[ m_nLen ] = '\0';
}



// Buffer Access

// Resize the character buffer so that it is at least nMinBufLength long and
// return a pointer to it for low level access. The buffer must be released
// with ReleaseBuffer before the next CString operation is beeing executed.
char* CString::GetBuffer( int nMinBufLength )
{
  ASSERT_VALID( this );

  m_buf.Resize( nMinBufLength + 1 );

  return m_buf;
}

// Same as GetBuffer with the exception that the buffer will be exact
// nLength long.
char* CString::GetBufferSetLength( int nLength )
{
  ASSERT_VALID( this );

  m_buf.ForceResize( nLength + 1 );
  
  return m_buf;
}

// Release the buffer previously locked with GetBuffer
void CString::ReleaseBuffer( int nNewLength )
{
  if ( nNewLength < 0 )
    m_nLen = strlen( (char *) m_buf );
  else
  {
    m_nLen = nNewLength;
    m_buf[ m_nLen ] = '\0';
  }

  ASSERT_VALID( this );
}



// Searching

int CString::Find( char ch, int nStart ) const
{
  ASSERT_VALID( this );

  char* pchFound = strchr( m_buf + nStart, ch );

  if ( pchFound )
    return pchFound - ( const char* ) m_buf;
  else
    return -1;
}

int CString::Find( const char* lpszSub, int nStart ) const
{
  ASSERT_VALID( this );

  char* pszFound = strstr( m_buf + nStart, lpszSub );

  if ( pszFound )
    return pszFound - ( const char* ) m_buf;
  else
    return -1;
}

int CString::ReverseFind( char ch ) const
{
  ASSERT_VALID( this );

  char* pchFound = strrchr( m_buf, ch );

  if ( pchFound )
    return pchFound - ( const char* ) m_buf;
  else
    return -1;
}

int CString::FindOneOf( const char* lpszCharSet ) const
{
  ASSERT_VALID( this );

  char* pszFound = strpbrk( m_buf, lpszCharSet );

  if ( pszFound )
    return pszFound - ( const char* ) m_buf;
  else
    return -1;
}



// Archive

CArchive& operator<<( CArchive& ar, const CString& string )
{
  ASSERT_VALID( &string );

  ar << string.m_nLen;
  ar.Write( ( const char* ) string.m_buf, string.m_nLen );

  return ar;
}

CArchive& operator>>( CArchive& ar, CString& string )
{
  ASSERT_VALID( &string );

  ar >> string.m_nLen;
  
  string.m_buf.Resize( string.m_nLen + 1 );
  ar.Read( ( char* ) string.m_buf, string.m_nLen );
  string.m_buf[ string.m_nLen ] = '\0';
  
  return ar;
}



// Streams

istream& operator>>( istream& rStream, CString& rString )
{
  ASSERT_VALID( &rString );

  // Let's hope the input will be no more than 1024 chars
  rString.m_buf.Realloc( 1025 );

  rStream >> rString.m_buf;
  rString.m_nLen = strlen( (char *) rString.m_buf );
  ASSERT( rString.m_nLen >= 0 && rString.m_nLen <= 1024 );
  
  rString.FreeExtra( );

  return rStream;
}



// Check class invariant (debug version only)
#ifndef NDEBUG
void CString::AssertValid( ) const
{
  ASSERT( m_buf.Size( ) >= m_nLen + 1 );
  ASSERT( m_nLen == ( int ) strlen( m_buf ) );
}
#endif
