//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: StrClass.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 10-13-97
// Last modified: 02-11-98
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This file contains the interface of the CString class. The interface is
// identical to the one of the CString class in MFC.

#ifndef __STRCLASS_H__
#define __STRCLASS_H__

#include <string.h>
#include <stdio.h>
#include <Decls.h>
#include <ctype.h>
#include <Buffer.h>

class CArchive;

class CString
{
public:
// Construction/Destruction
  CString( );
  // CString( const CString& stringSrc );
  CString( char ch, int nRepeat = 1 );
  CString( const char* lpch, int nLength );
  CString( const unsigned char* lpsz );
  CString( const char* lpsz );
  // ~CString( );

protected:
  CString( int nLen );

public:
// The String as an Array
  int GetLength( ) const;
  BOOL IsEmpty( ) const;
  void Empty( );
  char GetAt( int nIndex ) const;
  char operator[]( int nIndex ) const;
  void SetAt( int nIndex, char ch );
  operator const char*( ) const;

// Assignment/Concatenation
  // const CString& operator=( const CString& rOther );
  const CString& operator=( char ch );
  const CString& operator=( const unsigned char* lpsz );
  const CString& operator=( const char* lpsz );

  friend CString operator+( const CString& string1, const CString& string2 );
  friend CString operator+( const CString& string, char ch );
  friend CString operator+( char ch, const CString& string );
  friend CString operator+( const CString& string, const char* lpsz );
  friend CString operator+( const char* lpsz, const CString& string );

  const CString& operator+=( const CString& string ); 
  const CString& operator+=( char ch );
  const CString& operator+=( const char* lpsz );

// Comparision
  friend BOOL operator==( const CString& s1, const CString& s2 );
  friend BOOL operator==( const CString& s1, const char* s2 );
  friend BOOL operator==( const char* s1, const CString& s2 );
  
  friend BOOL operator!=( const CString& s1, const CString& s2 );
  friend BOOL operator!=( const CString& s1, const char* s2 );
  friend BOOL operator!=( const char* s1, const CString& s2);
  
  friend BOOL operator<( const CString& s1, const CString& s2 );
  friend BOOL operator<( const CString& s1, const char* s2 );
  friend BOOL operator<( const char* s1, const CString& s2 );
  
  friend BOOL operator>( const CString& s1, const CString& s2 );
  friend BOOL operator>( const CString& s1, const char* s2 );
  friend BOOL operator>( const char* s1, const CString& s2 );
  
  friend BOOL operator<=( const CString& s1, const CString& s2 );
  friend BOOL operator<=( const CString& s1, const char* s2 );
  friend BOOL operator<=( const char* s1, const CString& s2 );
  
  friend BOOL operator>=( const CString& s1, const CString& s2 );
  friend BOOL operator>=( const CString& s1, const char* s2 );
  friend BOOL operator>=( const char* s1, const CString& s2 );
  
  int Compare( const char* lpsz ) const; 
  int CompareNoCase( const char* lpsz ) const;
  int Collate( const char* lpsz ) const;

// Extraction
  CString Mid( int nFirst ) const;
  CString Mid( int nFirst, int nCount ) const;
  CString Left( int nCount ) const;
  CString Right( int nCount ) const;
  CString SpanIncluding( const char* lpszCharSet ) const;
  CString SpanExcluding( const char* lpszCharSet ) const;

// Other conversion
  void MakeUpper( );
  void MakeLower( );
  void MakeReverse( );
  void Format( const char* lpszFormat, ... );
  void TrimLeft( );
  void TrimRight( );

// Searching
  int Find( char ch, int nStart = 0 ) const;
  int Find( const char* lpszSub, int nStart = 0 ) const;
  int ReverseFind( char ch ) const;
  int FindOneOf( const char* lpszCharSet ) const;

// Archive
  friend CArchive& operator<<( CArchive& ar, const CString& string );
  friend CArchive& operator>>( CArchive& ar, CString& string );

// Buffer Access
  char* GetBuffer( int nMinBufLength );
  char* GetBufferSetLength( int nMinBufLength );
  void ReleaseBuffer( int nNewLength = -1 );
  void FreeExtra( );
  friend istream& operator>>( istream& rStream, CString& rString );

// Debug
#ifndef NDEBUG
  void AssertValid( ) const;
#endif

private:
  int m_nLen;
  CBuffer< char > m_buf;
};



// The String as an Array

inline int CString::GetLength( ) const
{
  ASSERT_VALID( this );

  return m_nLen;
}

inline BOOL CString::IsEmpty( ) const
{
  ASSERT_VALID( this );

  return m_nLen == 0;
}

inline void CString::Empty( )
{
  ASSERT_VALID( this );

  m_nLen = 0;
  m_buf.ForceResize( 1 );
  m_buf[ 0 ] = 0;
}

inline char CString::GetAt( int nIndex ) const
{
  ASSERT_VALID( this );
  ASSERT( nIndex >= 0 && nIndex < m_nLen );

  return m_buf[ nIndex ];
}

inline char CString::operator[]( int nIndex ) const
{
  ASSERT_VALID( this );
  ASSERT( nIndex >= 0 && nIndex < m_nLen );

  return GetAt( nIndex );
}

inline void CString::SetAt( int nIndex, char ch )
{
  ASSERT_VALID( this );
  ASSERT( nIndex >= 0 && nIndex < m_nLen );

  m_buf[ nIndex ] = ch;
}

inline CString::operator const char*( ) const
{
  ASSERT_VALID( this );

  return m_buf;
}



// Other conversions

inline void CString::MakeUpper( )
{
  ASSERT_VALID( this );

#ifdef _WIN32
  strupr( m_buf );
#else
  for ( int i = 0; i < m_nLen; i++ )
    m_buf[ i ] = toupper( ( int ) m_buf[ i ] );
#endif
}

inline void CString::MakeLower( )
{
  ASSERT_VALID( this );

#ifdef _WIN32
  strlwr( m_buf );
#else
  for ( int i = 0; i < m_nLen; i++ )
    m_buf[ i ] = tolower( ( int ) m_buf[ i ] );
#endif
}

inline void CString::MakeReverse( )
{
  ASSERT_VALID( this );

#ifdef _WIN32
  strrev( m_buf );
#else
  char chTemp;

  for ( int i = 0; i < m_nLen / 2; i++ )
  {
    chTemp = m_buf[ i ];
    m_buf[ i ] = m_buf[ m_nLen - 1 - i ];
    m_buf[ m_nLen - i - 1 ] = chTemp;
  }
#endif
}



// Comparison

inline int CString::Compare( const char* lpsz ) const
{
  ASSERT_VALID( this );

  return strcmp( m_buf, lpsz );
}

inline int CString::CompareNoCase( const char* lpsz ) const
{
  ASSERT_VALID( this );

#ifdef _WIN32
  return stricmp( m_buf, lpsz );
#else
  return strcasecmp( m_buf, lpsz );
#endif
}

inline int CString::Collate( const char* lpsz ) const
{
  ASSERT_VALID( this );

  return strcoll( m_buf, lpsz );
}

inline BOOL operator==( const CString& s1, const CString& s2 )
  {  return s1.Compare( s2 ) == 0; }
inline BOOL operator==( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) == 0; }
inline BOOL operator==( const char* s1, const CString& s2 )
  {  return s2.Compare( s1 ) == 0; }
inline BOOL operator!=( const CString& s1, const CString& s2 )
  {  return s1.Compare( s2 ) != 0; }
inline BOOL operator!=( const CString& s1, const char* s2 )
  {  return s1.Compare( s2 ) != 0; }
inline BOOL operator!=(const char* s1, const CString& s2)
  { return s2.Compare( s1 ) != 0; }
inline BOOL operator<( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) < 0; }
inline BOOL operator<( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) < 0; }
inline BOOL operator<( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) > 0; }
inline BOOL operator>( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) > 0; }
inline BOOL operator>( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) > 0; }
inline BOOL operator>( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) < 0; }
inline BOOL operator<=( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) <= 0; }
inline BOOL operator<=( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) <= 0; }
inline BOOL operator<=( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) >= 0; }
inline BOOL operator>=( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) >= 0; }
inline BOOL operator>=( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) >= 0; }
inline BOOL operator>=( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) <= 0; }



// Extraction

inline CString CString::Mid( int nFirst ) const
{
  ASSERT_VALID( this );
  ASSERT( nFirst >= 0 && nFirst < m_nLen || ( nFirst == 0 && m_nLen == 0 ) );

  return CString( m_buf + nFirst );
}

inline CString CString::Mid( int nFirst, int nCount ) const
{
  ASSERT_VALID( this );
  ASSERT( nFirst >= 0 && nFirst < m_nLen );
  ASSERT( nFirst + nCount <= m_nLen );

  return CString( m_buf + nFirst, nCount );
}

inline CString CString::Left( int nCount ) const
{
  ASSERT_VALID( this );
  ASSERT( nCount >= 0 && nCount <= m_nLen );

  return CString( m_buf, nCount );
}

inline CString CString::Right( int nCount ) const
{
  ASSERT_VALID( this );
  ASSERT( nCount >= 0 && nCount < m_nLen );

  return CString( m_buf + m_nLen - nCount, nCount );
}

inline CString CString::SpanIncluding( const char* lpszCharSet ) const
{
  ASSERT_VALID( this );

  return Left( strspn( m_buf, lpszCharSet) );
}

inline CString CString::SpanExcluding( const char* lpszCharSet ) const
{
  ASSERT_VALID( this );

  return Left( strcspn( m_buf, lpszCharSet ) );
}



// Buffer Access

inline void CString::FreeExtra( )
{
  ASSERT_VALID( this );

  m_buf.ForceResize( m_nLen + 1 );
}

#endif   // __STRCLASS_H__
