////////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: TreeBuffer.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 11-21-97
// Last modified: 11-22-97
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// Low-level general tree class with optimized memory management. This class is
// used to build high-level classes like linked lists, binary trees, quad trees
// and so on. It provides the speed of a static implementation and the
// flexibility of a dynamic implementation.

#ifndef __TREEBUFFER_H__
#define __TREEBUFFER_H__

#include <Buffer.h>

template< class TYPE, int DEGREE >
class CTreeBuffer
{
public:
  class CTreeBufferNode
  {
  public:
    TYPE m_value;
    CTreeBufferNode* m_ppRelatives[ DEGREE ];
  };

private:
  class CBlock   // warning: variable length object
  {
  public:
    CBlock* m_pNext;
    CTreeBufferNode m_pNodes[ 1 ];   
  }; 
  // trick: node pointer that points right behind m_pNext. Some compilers do
  // not accept m_pNodes[ 0 ] so use m_pNodes[ 1 ]

public:
  // Construction / Destruction
  CTreeBuffer( int nBlockSize = 8 );
  ~CTreeBuffer( );
  
  // Resizing
  CTreeBufferNode* NewNode( );
  void DeleteNode( CTreeBufferNode* pNode );
 
private:
  int m_nBlockSize;   // number of elements a block contains
  CBlock* m_pHead;   // the linked memory blocks
  CTreeBufferNode* m_pFreeNodes;   // singly linked list of free nodes
};



// Construction / Destruction
  
template< class TYPE, int DEGREE >
CTreeBuffer< TYPE, DEGREE >::CTreeBuffer( int nBlockSize )
{
  m_nBlockSize = nBlockSize;
  m_pHead = NULL;
  m_pFreeNodes = NULL;
}

template< class TYPE, int DEGREE >
CTreeBuffer< TYPE, DEGREE >::~CTreeBuffer( )
{
  CBlock* pNext;
  CBlock* pBlock = m_pHead;

  while ( pBlock != NULL )
  {
    pNext = pBlock->m_pNext;
    DestructElements( pBlock->m_pNodes, m_nBlockSize );
    delete [ ] ( BYTE* ) pBlock;
    pBlock = pNext;
  }
}



// Resizing

template< class TYPE, int DEGREE >
CTreeBuffer< TYPE, DEGREE >::CTreeBufferNode* CTreeBuffer< TYPE, DEGREE >::NewNode( )
{
  if ( m_pFreeNodes == NULL )
  {
    // we declared m_pNodes[ 1 ] => substract that node from m_nBlockSize
    CBlock* pBlock = ( CBlock* ) new BYTE[ sizeof( CBlock ) +
      ( sizeof( CTreeBufferNode ) * ( m_nBlockSize - 1 ) ) ];
    ConstructElements( pBlock->m_pNodes, m_nBlockSize );

    // add in reverse order for simplicity
    pBlock->m_pNext = m_pHead;
    m_pHead = pBlock;

    // add nodes to list of free nodes (in reverse order)
    for ( int i = 0; i < m_nBlockSize; i++ )
    {
      *pBlock->m_pNodes[ i ].m_ppRelatives = m_pFreeNodes;
      m_pFreeNodes = pBlock->m_pNodes + i ;
    }
  }

  ASSERT( m_pFreeNodes != NULL );

  // Get first node in the list of free nodes
  CTreeBufferNode* pNewNode = m_pFreeNodes;
  m_pFreeNodes = *m_pFreeNodes->m_ppRelatives;

  // Set relatives to NULL
  memset( pNewNode->m_ppRelatives, 0, sizeof( CTreeBufferNode* ) * DEGREE );

  return pNewNode;
}

template< class TYPE, int DEGREE >
void CTreeBuffer< TYPE, DEGREE >::DeleteNode( CTreeBufferNode* pNode )
{
  // add node to the list of free nodes
  *pNode->m_ppRelatives = m_pFreeNodes;
  m_pFreeNodes = pNode;
}

#endif   // __TREEBUFFER_H__
